<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\TreeAccountingCollection;
use App\Http\Resources\Accounting\TreeAccountingLangResource;
use App\Http\Resources\Accounting\TreeAccountingResource;
use App\Http\Resources\Accounting\TreeAccountingSelectResource;
use App\sys\ApiResponse;
use App\sys\Helper;
use App\sys\Services\Accounting\TreeAccountingServices;
use App\sys\Services\Accounting\TreeAccountingTranslationServices;
use Illuminate\Http\Request;

class TreeAccountingController extends Controller
{
    use ApiResponse;

    private $tree;

    public function __construct()
    {
        $this->tree = new TreeAccountingServices;
        $this->treeTranslations = new TreeAccountingTranslationServices;
    }

    public function getByCurrency($id)
    {
        $data = $this->tree->gettreeByCurrency($id);

        return $this->apiResponse(200, 'getByCurrency', null, TreeAccountingSelectResource::collection($data));
    }

    public function getBanksByCurrency($id)
    {
        $data = $this->tree->getBanksByCurrency($id);

        return $this->apiResponse(200, 'getBanksByCurrency', null, TreeAccountingSelectResource::collection($data));
    }

    public function getSerialNumber($id = 0)
    {
        $result = Helper::generateNextSerial($id, null);

        if (is_array($result) && array_key_exists('status', $result) && $result['status'] === false) {
            $message = $result['message'] ?? 'Serial generation failed';

            return $this->apiResponse(400, $message, $result['errors'] ?? [$message], null);
        }

        return $this->apiResponse(200, 'تم', null, $result);
    }

    public function validateSerial(Request $request)
    {
        $request->validate([
            'parent_id' => 'nullable|integer',
            'accounts_structure_id' => 'required|integer',
            'serial_candidate' => 'required|string',
        ]);

        $parentId = $request->input('parent_id', 0);
        $levelId = (int) $request->input('accounts_structure_id');
        $candidate = preg_replace('/[^0-9]/', '', $request->input('serial_candidate'));

        $isUnique = Helper::isSerialUnique($parentId, $levelId, $candidate);

        return $this->apiResponse(200, 'serial number validation success', null, ['isUnique' => $isUnique]);

    }

    public function create(Request $request)
    {
        if ($data = $this->tree->create($request->all())) {
            // optional: bootstrap empty translation for default language if needed
            return $this->apiResponse(200, 'تم إنشاء الحساب بنجاح', null, $data);
        }
        $errors = $this->tree->errors();

        return $this->apiResponse(400, 'errors', $errors, null);
    }

    public function saveTranslation($id, Request $request)
    {
        if (! $this->treeTranslations->save($request->all(), (int) $id)) {
            $errors = $this->treeTranslations->errors();

            return $this->apiResponse(400, 'errors', $errors, null);
        }

        return $this->apiResponse(200, 'تم', null, null);
    }

    public function getTranslation($id)
    {
        $row = $this->tree->findById((int) $id);
        if (! $row) {
            return $this->apiResponse(404, 'غير موجود', ['not found'], null);
        }

        return $this->apiResponse(200, 'تم', null, new TreeAccountingLangResource($row));
    }

    public function delLang($id)
    {
        if ($this->treeTranslations->del((int) $id)) {
            return $this->apiResponse(200, 'تم الحذف', null, true);
        }
        $errors = $this->treeTranslations->errors();

        return $this->apiResponse(400, 'errors', $errors, null);
    }

    public function updateBasic(Request $request)
    {
        if ($data = $this->tree->updateBasic($request->all())) {
            return $this->apiResponse(200, 'updated :) ', null, $data);
        }
        $errors = $this->tree->errors();

        return $this->apiResponse(400, 'errors', $errors, null);
    }

    public function parents()
    {
        $data = $this->tree->getParents();

        return $this->apiResponse(200, 'تم', null, TreeAccountingResource::collection($data));
    }

    public function active()
    {
        $data = $this->tree->getActive();

        return $this->apiResponse(200, 'تم', null, TreeAccountingResource::collection($data));
    }

    public function byParent($parentId)
    {
        $data = $this->tree->getByParentId((int) $parentId);

        return $this->apiResponse(200, 'تم', null, TreeAccountingResource::collection($data));
    }

    public function list()
    {
        $data = $this->tree->getPaginated();

        return $this->apiResponse(200, 'تم', null, (new TreeAccountingCollection($data))->toArray(request()));
    }

    public function show($id)
    {
        $data = $this->tree->findById((int) $id);
        if (! $data) {
            return $this->apiResponse(404, 'غير موجود', ['not found'], null);
        }
        $data->load('translations');

        return $this->apiResponse(200, 'تم', null, new TreeAccountingResource($data));
    }

    public function getAll()
    {
        $data = $this->tree->getAll();

        return $this->apiResponse(200, 'تم', null, (new TreeAccountingCollection($data))->toArray(request()));

    }
}
