<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\AreaCollection;
use App\Http\Resources\AreaResource;
use App\Http\Resources\General\AreaHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\AreaService;
use App\sys\Services\General\AreaTranslationService;
use Illuminate\Http\Request;

class AreaController extends Controller
{
    use ApiResponse;

    protected AreaService $areaService;

    protected AreaTranslationService $areaTranslationService;

    public function __construct(AreaService $areaService)
    {
        $this->areaService = $areaService;
        $this->areaTranslationService = new AreaTranslationService;
    }

    public function index()
    {
        $areas = $this->areaService->getPaginatedAreas();

        return $this->apiResponse(200, 'success', null, new AreaCollection($areas));
    }

    public function show(int $id)
    {
        $area = $this->areaService->getAreaById($id);
        if (! $area) {
            $errors = $this->areaService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AreaResource($area));
    }

    public function store(Request $request)
    {
        $response = $this->areaService->createArea($request->all());
        if (! $response) {
            $errors = $this->areaService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AreaResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->areaService->updateArea($request->all());
        if (! $response) {
            $errors = $this->areaService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AreaResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->areaService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Areas deleted successfully', null, null);
        }
        $errors = $this->areaService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->areaService->getActive();
        $collect = new \App\Http\Resources\AreaCollection($data);

        return $this->apiResponse(200, 'Active Areas retrieved successfully', null, $collect);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->areaTranslationService->save($request->all(), $id)) {
            $errors = $this->areaTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->areaService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\Area\AreaLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->areaService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->areaTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->areaTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getByCity($cityId)
    {
        $areas = $this->areaService->getAreasByCity($cityId);
        if (! $areas) {
            $errors = $this->areaService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AreaCollection($areas));
    }

    public function getAllChanges($id)
    {
        $area = $this->areaService->getAreaById($id);
        if (! $area) {
            $errors = $this->areaService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->areaService->getAllChanges($area);

        return $this->apiResponse(200, 'success', null, AreaHistoryResource::collection($changes));
    }
}
