<?php

namespace App\Models\General;

use App\Models\Suppliers\Suppliers;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Auditable as AuditableTrait;
use OwenIt\Auditing\Contracts\Auditable;

class Service extends Model implements Auditable
{
    use AuditableTrait, HasFactory, SoftDeletes;

    protected $table = 'services';

    const CREATED_AT = 'created';

    const UPDATED_AT = 'modified';

    protected $fillable = [
        'travel_tourism_type',
        'title',
        'category_id',
        'item_code',
        'unit_type',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Attributes to audit
     */
    protected $auditInclude = [
        'travel_tourism_type',
        'title',
        'category_id',
        'item_code',
        'unit_type',
        'is_active',
    ];

    /**
     * Attributes to exclude from audit
     */
    protected $auditExclude = [
        'created',
        'modified',
    ];

    /**
     * Audit events to record
     */
    protected $auditEvents = [
        'updated',
    ];

    public function currencies()
    {
        return $this->belongsToMany(Currency::class, 'services_currencies', 'service_id', 'currency_id');
    }

    public function servicesCurrencies()
    {
        return $this->hasMany(ServicesCurrencies::class, 'service_id');
    }

    public function translations()
    {
        return $this->hasMany(ServicesTranslation::class, 'services_id');
    }

    public function currentTranslation()
    {
        return $this->hasOne(ServicesTranslation::class, 'services_id')
            ->where('lang_id', app('lang_id'));
    }

    public function suppliers()
    {
        return $this->belongsToMany(Suppliers::class, 'suppliers_services', 'service_id', 'supplier_id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function users()
    {
        return $this->belongsToMany(User::class, 'service_user', 'service_id', 'user_id')->withPivot('type');
    }

    public function getCurrencyWithTree($currencyId = null)
    {
        $query = $this->servicesCurrencies()->with(['currency', 'treeAccounting', 'treeIncomeAccounting']);

        if ($currencyId) {
            $query->where('currency_id', $currencyId);
        }

        return $query->get()->map(function ($serviceCurrency) {
            return [
                'id' => $serviceCurrency->id,
                'currency_id' => $serviceCurrency->currency_id,
                'currency_name' => $serviceCurrency->currency->name,
                'tree_accounting' => [
                    'id' => $serviceCurrency->treeAccounting->tree_accounting_id,
                    'name' => $serviceCurrency->treeAccounting->title,
                ],
                'tree_income_accounting' => [
                    'id' => $serviceCurrency->treeIncomeAccounting->tree_accounting_id,
                    'name' => $serviceCurrency->treeIncomeAccounting->title,
                ],
            ];
        });
    }
}
