<?php

namespace App\Models\General;

use App\Models\Accounting\TreeAccounting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ServicesCurrencies extends Model
{
    use HasFactory;

    protected $table = 'services_currencies';

    protected $fillable = [
        'service_id',
        'currency_id',
        'tree_accounting_id',
        'tree_income_accounting_id',
    ];

    public function service()
    {
        return $this->belongsTo(Service::class, 'service_id');
    }

    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_id');
    }

    public function treeAccounting()
    {
        return $this->belongsTo(TreeAccounting::class, 'tree_accounting_id', 'tree_accounting_id');
    }

    public function treeIncomeAccounting()
    {
        return $this->belongsTo(TreeAccounting::class, 'tree_income_accounting_id', 'tree_accounting_id');
    }

    public static function rules()
    {
        return [
            'service_id' => 'required|integer|exists:services,id',
            'currency_id' => 'required|integer|exists:currencies,id,status,1',
            'tree_accounting_id' => 'required|integer|exists:tree_accounting,tree_accounting_id',
            'tree_income_accounting_id' => 'required|integer|exists:tree_accounting,tree_accounting_id',
        ];
    }

    public static function uniqueValidationRules($serviceId, $excludeId = null)
    {
        $rules = [
            'service_id' => 'required|integer|exists:services,id',
            'currency_id' => [
                'required',
                'integer',
                'exists:currencies,id,status,1',
                function ($attribute, $value, $fail) use ($serviceId, $excludeId) {
                    $query = static::where('service_id', $serviceId)
                        ->where('currency_id', $value);

                    if ($excludeId) {
                        $query->where('id', '!=', $excludeId);
                    }

                    if ($query->exists()) {
                        $fail('This currency is already assigned to this service.');
                    }
                },
            ],
            'tree_accounting_id' => 'required|integer|exists:tree_accounting,tree_accounting_id',
            'tree_income_accounting_id' => 'required|integer|exists:tree_accounting,tree_accounting_id',
        ];

        return $rules;
    }
}
