<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\CostCenter;
use App\Models\Accounting\Invoice;
use App\Models\Accounting\Transfer;
use App\Models\Accounting\TreeAccounting;
use App\Models\General\CompanyAccountMappings;
use App\Models\General\Currency;
use App\Models\UserAccountMapping;
use Illuminate\Support\Facades\DB;

class ReportRepository
{
    public function getReports()
    {
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $transfers = Transfer::with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->whereDate('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            })->get();

        return $this->formatTransferResponse($transfers, $currency, $defaultCurrency);
    }

    public function getBalanceAndIncomeReport($reportType)
    {
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        if ($reportType === 'balance') {
            $types = ['assets', 'liabilities'];
        } elseif ($reportType === 'income') {
            $types = ['expenditure', 'revenue'];
        }

        $treeAccounts = TreeAccounting::whereIn('id', function ($query) use ($types) {
            $query->select('tree_account_id')
                ->from('tree_account_settings')
                ->whereIn('type', $types);
        })
            ->with('getAllChildren')
            ->get();

        $treeAccountingIds = [];

        foreach ($treeAccounts as $account) {
            $treeAccountingIds[] = $account->id;
            $treeAccountingIds = array_merge($treeAccountingIds, $account->getAllChildrenIdsFromRelationship());
        }
        $treeAccountingIds = array_values(array_unique($treeAccountingIds));

        $transfers = Transfer::whereIn('tree_accounting_id', $treeAccountingIds)->with('constraint', 'treeAccounting.currentTranslation', 'currency', 'costCenter')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->where('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->where('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            })
            ->get();

        return $this->formatTransferResponse($transfers, $currency, $defaultCurrency);
    }

    private function formatTransferResponse($transfers, $currency, $defaultCurrency)
    {
        if ($currency == null) {
            return [
                'data' => $transfers->map(function ($transfer) {
                    return [
                        'id' => $transfer->id,
                        'constraint_id' => $transfer->constraint->id ?? null,
                        'constraint_number_doc' => $transfer->constraint->number_doc ?? null,
                        'transfer_name' => $transfer->name,
                        'creditor' => $transfer->creditor,
                        'debit' => $transfer->debit,
                        'creditor_other' => $transfer->currency_creditor,
                        'debit_other' => $transfer->currency_debit,
                        'date' => $transfer->date,
                        'type' => $transfer->constraint->capture_exchange ?? null,
                        'constraint_name' => $transfer->constraint->name ?? null,
                        'constraint_description' => $transfer->constraint->description ?? null,
                        'tree_accounting_title' => $transfer->treeAccounting->currentTranslation->title ?? ($transfer->treeAccounting->title ?? null),
                        'tree_accounting_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                        'currency_name' => $transfer->currency->name ?? null,
                        'currency_code' => $transfer->currency->code ?? null,
                        'currency_symbol' => $transfer->currency->symbol ?? null,
                        'currency_transfer_rate' => $transfer->currency_transfer_rate ?? null,
                        'cost_center_name' => $this->costCenter->name ?? null,
                    ];
                }),
                'summery' => [
                    'total_debit' => $transfers->sum('debit'),
                    'total_creditor' => $transfers->sum('creditor'),
                ],
            ];
        } else {
            $totalCreditAfterExchange = $transfers->sum(function ($transfer) {
                return $transfer->currency_creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitAfterExchange = $transfers->sum(function ($transfer) {
                return $transfer->currency_debit * ($transfer->currency_transfer_rate ?? 1);
            });

            return [
                'data' => $transfers->map(function ($transfer) use ($defaultCurrency) {
                    return [
                        'id' => $transfer->id,
                        'constraint_id' => $transfer->constraint->id ?? null,
                        'constraint_number_doc' => $transfer->constraint->number_doc ?? null,
                        'transfer_name' => $transfer->name,
                        'creditor' => $transfer->currency_creditor,
                        'debit' => $transfer->currency_debit,
                        'creditor_other' => $transfer->creditor,
                        'debit_other' => $transfer->debit,
                        'date' => $transfer->date,
                        'type' => $transfer->constraint->capture_exchange ?? null,
                        'constraint_name' => $transfer->constraint->name ?? null,
                        'constraint_description' => $transfer->constraint->description ?? null,
                        'tree_accounting_title' => $transfer->treeAccounting->currentTranslation->title ?? ($transfer->treeAccounting->title ?? null),
                        'tree_accounting_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                        'currency_name' => $transfer->currency->name ?? null,
                        'currency_code' => $transfer->currency->code ?? null,
                        'currency_symbol' => $transfer->currency->symbol ?? null,
                        'currency_transfer_rate' => $transfer->currency_transfer_rate ?? null,
                        'default_currency_name' => $defaultCurrency->name ?? null,
                        'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                        'default_currency_code' => $defaultCurrency->code ?? null,
                        'default_currency_exchange_rate' => $defaultCurrency->exchange_rate ?? null,
                        'cost_center_name' => $this->costCenter->name ?? null,
                    ];
                }),
                'summery' => [
                    'total_debit' => $totalDebitAfterExchange,
                    'total_creditor' => $totalCreditAfterExchange,
                ],
            ];
        }
    }

    public function getHierarchicalTreeAccountReport($currency, $companyIds, $dateFrom, $dateTo, $type)
    {

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $types = ($type == 'balance') ? ['assets', 'liabilities'] : ['revenue', 'expenditure'];

        $mainTreeAccounts = TreeAccounting::where('tree_accounting_id', 0)
            ->where('active', 1)
            ->whereIn('id', function ($query) use ($types) {
                $query->select('tree_account_id')
                    ->from('tree_account_settings')
                    ->whereIn('type', $types);
            })
            ->with(['getAllChildren'])
            ->get();

        $result = [];
        $grandTotalBalance = 0;
        $totalRevenue = 0;
        $totalExpenditure = 0;
        $totalProfitsLosses = 0;

        foreach ($mainTreeAccounts as $mainAccount) {
            $accountType = $mainAccount->settings()->whereIn('type', $types)->value('type');
            $accountData = $this->buildHierarchicalAccountData(
                $mainAccount,
                $currency,
                $companyIds,
                $dateFrom,
                $dateTo,
                $defaultCurrency,
                $accountType,
                $type
            );

            if ($accountData) {
                // Get the account type from tree_account_settings

                // Adjust grand total based on type
                if ($type == 'balance') {
                    if ($accountType == 'assets') {
                        $grandTotalBalance += $accountData['balance'];
                    } elseif ($accountType == 'liabilities') {
                        $grandTotalBalance -= $accountData['balance'];
                    }

                } else {
                    if ($accountType == 'revenue') {
                        $totalRevenue = $accountData['balance'];
                        $grandTotalBalance += $accountData['balance'];
                    } elseif ($accountType == 'expenditure') {
                        $totalExpenditure = $accountData['balance'];
                        $grandTotalBalance -= $accountData['balance'];
                    }
                }

                $result[] = $accountData;
            }
        }
        if ($type == 'balance') {
            $income = $this->getHierarchicalTreeAccountReport($currency, $companyIds, $dateFrom, $dateTo, 'income');
            $totalProfitsLosses = $income['income']['totalRevenue'] ?? 0 - $income['income']['totalExpenditure'] ?? 0;
        }

        return [
            'data' => $result,
            'income' => [
                'totalRevenue' => $totalRevenue,
                'totalExpenditure' => $totalExpenditure,
            ],
            'balance' => [
                'totalProfitsLosses' => $totalProfitsLosses,
            ],
            'totals' => [
                'grand_total_balance' => $grandTotalBalance,
            ],
            'currency_info' => $currency ? null : [
                'default_currency_name' => $defaultCurrency->name ?? null,
                'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                'default_currency_code' => $defaultCurrency->code ?? null,
            ],
        ];
    }

    private function buildHierarchicalAccountData($account, $currency, $companyIds, $dateFrom, $dateTo, $defaultCurrency, $accountType = null, $type = null)
    {
        // Get all child account IDs for this account
        $allChildIds = [$account->id];
        if ($account->getAllChildren) {
            $allChildIds = array_merge($allChildIds, $account->getAllChildrenIdsFromRelationship());
        }

        // Get transfers for this account and all its children
        $transfersQuery = Transfer::whereIn('tree_accounting_id', $allChildIds)
            ->with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->whereDate('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            });

        $transfers = $transfersQuery->get();

        // Calculate totals
        $totalCreditor = 0;
        $totalDebit = 0;

        if ($currency) {
            // If currency is specified, use direct amounts
            $totalCreditor = $transfers->sum('creditor');
            $totalDebit = $transfers->sum('debit');
        } else {
            // If no currency specified, convert to default currency
            $totalCreditor = $transfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebit = $transfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        // Build children data
        $children = [];
        if ($account->getAllChildren) {
            foreach ($account->getAllChildren as $child) {
                $childData = $this->buildHierarchicalAccountData(
                    $child,
                    $currency,
                    $companyIds,
                    $dateFrom,
                    $dateTo,
                    $defaultCurrency,
                    $accountType, $type
                );

                if ($childData) {
                    $children[] = $childData;
                }
            }
        }

        // Calculate total from children
        $childrenTotalCreditor = array_sum(array_column($children, 'total_creditor'));
        $childrenTotalDebit = array_sum(array_column($children, 'total_debit'));

        // If this account has no direct transfers but has children, use children totals
        if ($transfers->isEmpty() && ! empty($children)) {
            $totalCreditor = $childrenTotalCreditor;
            $totalDebit = $childrenTotalDebit;
        }

        $balance = $accountType == 'revenue' ? $totalCreditor - $totalDebit : $totalDebit - $totalCreditor;

        return [
            'id' => $account->id,
            'title' => $account->title,
            'the_level' => $account->the_level,
            'serial_number' => $account->serial_number,
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'balance' => $balance, // Debit - Credit
            'children' => $children,
            'has_children' => ! empty($children),
            'direct_transfers_count' => $transfers->count(),
        ];
    }

    public function getHierarchicalTreeAccountReportTrailBalance()
    {
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from', '2025-01-01');
        $dateTo = request('date_to');

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $types = ['assets', 'liabilities', 'revenue', 'expenditure'];

        $mainTreeAccounts = TreeAccounting::where('tree_accounting_id', 0)
            ->where('active', 1)
            ->with(['getAllChildren'])
            ->get();

        $result = [];
        $grandTotals = [
            'total_creditor_initial' => 0,
            'total_debit_initial' => 0, // افتاحي
            'total_creditor_within' => 0,  // بالمجميع
            'total_debit_within' => 0,
            'total_creditor' => 0, // اجمالي
            'total_debit' => 0,
            'closing_creditor' => 0,
            'closing_debit' => 0,
            'balance_initial' => 0,
            'balance_within' => 0,
            'balance_total' => 0,
        ];

        foreach ($mainTreeAccounts as $mainAccount) {
            $accountData = $this->buildHierarchicalAccountDataWithInitial(
                $mainAccount,
                $currency,
                $companyIds,
                $dateFrom,
                $dateTo,
                $defaultCurrency
            );

            if ($accountData) {
                // Get the account type from tree_account_settings
                $accountType = $mainAccount->settings()->whereIn('type', $types)->value('type');
                $accountData['account_type'] = $accountType;

                // Add to grand totals
                $grandTotals['total_creditor_initial'] += $accountData['total_creditor_initial'];
                $grandTotals['total_debit_initial'] += $accountData['total_debit_initial'];
                $grandTotals['total_creditor_within'] += $accountData['total_creditor_within'];
                $grandTotals['total_debit_within'] += $accountData['total_debit_within'];
                $grandTotals['total_creditor'] += $accountData['total_creditor'];
                $grandTotals['total_debit'] += $accountData['total_debit'];

                if ($accountData['total_creditor'] >= $accountData['total_debit']) {
                    $grandTotals['closing_creditor'] += $accountData['total_creditor'] - $accountData['total_debit'];
                    $grandTotals['closing_debit'] = 0;
                } else {
                    $grandTotals['closing_debit'] += $accountData['total_debit'] - $accountData['total_creditor'];
                    $grandTotals['closing_creditor'] = 0;
                }
                // Calculate balances based on account type
                if (in_array($accountType, ['assets', 'expenditure'])) {
                    $grandTotals['balance_initial'] += $accountData['balance_initial'];
                    $grandTotals['balance_within'] += $accountData['balance_within'];
                    $grandTotals['balance_total'] += $accountData['balance_total'];
                } else {
                    $grandTotals['balance_initial'] -= $accountData['balance_initial'];
                    $grandTotals['balance_within'] -= $accountData['balance_within'];
                    $grandTotals['balance_total'] -= $accountData['balance_total'];
                }

                $result[] = $accountData;
            }
        }

        usort($result, function ($a, $b) {
            return $a['serial_number'] <=> $b['serial_number'];
        });

        return [
            'data' => $result,
            'totals' => $grandTotals,
            'currency_info' => $currency ? null : [
                'default_currency_name' => $defaultCurrency->name ?? null,
                'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                'default_currency_code' => $defaultCurrency->code ?? null,
            ],
            'period_info' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
        ];
    }

    private function buildHierarchicalAccountDataWithInitial($account, $currency, $companyIds, $dateFrom, $dateTo, $defaultCurrency)
    {
        $allChildIds = [$account->id];
        if ($account->getAllChildren) {
            $allChildIds = array_merge($allChildIds, $account->getAllChildrenIdsFromRelationship());
        }

        $baseTransfersQuery = Transfer::whereIn('tree_accounting_id', $allChildIds)
            ->with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            });

        $initialTransfersQuery = clone $baseTransfersQuery;
        $initialTransfers = $initialTransfersQuery->whereDate('date', '<', $dateFrom)->get();

        $withinTransfersQuery = clone $baseTransfersQuery;
        $withinTransfers = $withinTransfersQuery
            ->whereDate('date', '>=', $dateFrom)
            ->whereDate('date', '<=', $dateTo)
            ->get();

        $totalCreditorInitial = 0;
        $totalDebitInitial = 0;

        if ($currency) {
            $totalCreditorInitial = $initialTransfers->sum('creditor');
            $totalDebitInitial = $initialTransfers->sum('debit');
        } else {
            $totalCreditorInitial = $initialTransfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitInitial = $initialTransfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        $totalCreditorWithin = 0;
        $totalDebitWithin = 0;

        if ($currency) {
            $totalCreditorWithin = $withinTransfers->sum('creditor');
            $totalDebitWithin = $withinTransfers->sum('debit');
        } else {
            $totalCreditorWithin = $withinTransfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitWithin = $withinTransfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        $children = [];
        if ($account->getAllChildren) {
            foreach ($account->getAllChildren as $child) {
                $childData = $this->buildHierarchicalAccountDataWithInitial(
                    $child,
                    $currency,
                    $companyIds,
                    $dateFrom,
                    $dateTo,
                    $defaultCurrency
                );

                if ($childData) {
                    $children[] = $childData;
                }
            }
        }

        $childrenCreditorInitial = array_sum(array_column($children, 'total_creditor_initial'));
        $childrenDebitInitial = array_sum(array_column($children, 'total_debit_initial'));
        $childrenCreditorWithin = array_sum(array_column($children, 'total_creditor_within'));
        $childrenDebitWithin = array_sum(array_column($children, 'total_debit_within'));

        if ($initialTransfers->isEmpty() && $withinTransfers->isEmpty() && ! empty($children)) {
            $totalCreditorInitial = $childrenCreditorInitial;
            $totalDebitInitial = $childrenDebitInitial;
            $totalCreditorWithin = $childrenCreditorWithin;
            $totalDebitWithin = $childrenDebitWithin;
        }

        $totalCreditor = $totalCreditorInitial + $totalCreditorWithin;
        $totalDebit = $totalDebitInitial + $totalDebitWithin;
        if ($totalCreditor >= $totalDebit) {
            $closing_creditor = $totalCreditor - $totalDebit;
            $closing_debit = 0;
        } else {
            $closing_debit = $totalDebit - $totalCreditor;
            $closing_creditor = 0;
        }

        $balanceInitial = $totalDebitInitial - $totalCreditorInitial;
        $balanceWithin = $totalDebitWithin - $totalCreditorWithin;
        $balanceTotal = $totalDebit - $totalCreditor;

        return [
            'id' => $account->id,
            'title' => $account->title,
            'the_level' => $account->the_level,
            'serial_number' => $account->serial_number,

            // Initial amounts (before start date)
            'total_creditor_initial' => $totalCreditorInitial,
            'total_debit_initial' => $totalDebitInitial,
            'balance_initial' => $balanceInitial,

            // Within period amounts
            'total_creditor_within' => $totalCreditorWithin,
            'total_debit_within' => $totalDebitWithin,
            'balance_within' => $balanceWithin,

            // Total amounts (initial + within)
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'closing_creditor' => $closing_creditor,
            'closing_debit' => $closing_debit,
            'balance_total' => $balanceTotal,

            'children' => $children,
            'has_children' => ! empty($children),
            'initial_transfers_count' => $initialTransfers->count(),
            'within_transfers_count' => $withinTransfers->count(),
            'total_transfers_count' => $initialTransfers->count() + $withinTransfers->count(),
        ];
    }

    public function treeAccountTransfer(array $request)
    {

        $trees = TreeAccounting::whereIn('id', $request['treeAccount_ids'] ?? [])
            ->with([
                'currentTranslation',
                'transfers' => function ($query) use ($request) {
                    $query->when($request['date_from'] ?? null, function ($query) use ($request) {
                        return $query->where('date', '>=', $request['date_from']);
                    })
                        ->when($request['date_to'] ?? null, function ($query) use ($request) {
                            return $query->where('date', '<=', $request['date_to']);
                        })
                        ->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                            $query->whereHas('constraint', function ($subQuery) use ($request) {
                                $subQuery->whereIn('company_id', $request['company_ids']);
                            });
                        });
                },
                'transfers.constraint',
                'transfers.costCenter',
                'transfers.currency.currentTranslation',
                'transfers.treeAccounting',
            ])->get();

        return $trees->map(function ($tree) use ($request) {
            $account['currency_id'] = $request['show_by_default'] ? null : $tree->currency_id;
            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;
            $totalBalance = 0;

            foreach ($tree->transfers as $transfer) {

                $debit = $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                $debit_other = $account['currency_id'] != null ? $transfer->debit ?? 0 : $transfer->currency_debit ?? 0;
                $credit = $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                $creditor_other = $account['currency_id'] != null ? $transfer->creditor ?? 0 : $transfer->currency_creditor ?? 0;
                $balance = $debit - $credit;
                $totalBalance += $balance;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'cost_center_name' => $transfer->costCenter->name ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'tree_account_title' => $transfer->treeAccounting->title ?? null,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'debit' => $debit,
                    'credit' => $credit,
                    'creditor_other' => $creditor_other,
                    'debit_other' => $debit_other,
                    'balance' => $totalBalance,
                    'rate' => $transfer->currency_transfer_rate,
                    'running_balance' => $totalBalance,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'tree_account_id' => $tree->id,
                'tree_account_title' => $tree->title,
                'tree_account_serial_number' => $tree->serial_number,
                'transactions' => $transactions,
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'total_balance' => $totalDebit - $totalCredit,
                'show_by_default' => $account['currency_id'] == null ? true : false,

            ];
        });
    }

    public function treeAccountTransferWithInitialBalance(array $request)
    {

        $trees = TreeAccounting::whereIn('id', $request['treeAccount_ids'] ?? [])
            ->with([
                'currentTranslation',
                'transfers' => function ($query) use ($request) {
                    $query->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                        $query->whereHas('constraint', function ($subQuery) use ($request) {
                            $subQuery->whereIn('company_id', $request['company_ids']);
                        });
                    });
                },
                'transfers.constraint',
                'transfers.costCenter',
                'transfers.currency.currentTranslation',
            ])
            ->get();

        return $trees->map(function ($tree) use ($request) {

            $account['currency_id'] = $request['show_by_default'] ? null : $tree->currency_id;

            // Calculate initial balance from transfers before start date
            $initialDebit = 0;
            $initialCredit = 0;

            if (! empty($request['date_from'])) {
                foreach ($tree->transfers as $transfer) {
                    if ($transfer->date < $request['date_from']) {
                        $initialDebit += $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                        $initialCredit += $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                    }
                }
            }

            $initialBalance = $initialDebit - $initialCredit;
            $runningBalance = $initialBalance;

            // Filter transfers within the date range and process transactions
            $transfersInRange = $tree->transfers->filter(function ($transfer) use ($request) {
                $withinDateFrom = empty($request['date_from']) || $transfer->date >= $request['date_from'];
                $withinDateTo = empty($request['date_to']) || $transfer->date <= $request['date_to'];

                return $withinDateFrom && $withinDateTo;
            });

            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;

            foreach ($transfersInRange as $transfer) {
                $debit = $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                $debit_other = $account['currency_id'] != null ? $transfer->debit ?? 0 : $transfer->currency_debit ?? 0;
                $credit = $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                $creditor_other = $account['currency_id'] != null ? $transfer->creditor ?? 0 : $transfer->currency_creditor ?? 0;
                $transferBalance = $debit - $credit;

                // Update running balance
                $runningBalance += $transferBalance;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'cost_center_name' => $transfer->costCenter->name ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'tree_account_title' => $transfer->treeAccounting->title ?? null,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'debit' => $debit,
                    'credit' => $credit,
                    'creditor_other' => $creditor_other,
                    'debit_other' => $debit_other,
                    'transfer_balance' => $transferBalance,
                    'running_balance' => $runningBalance,
                    'rate' => $transfer->currency_transfer_rate,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'tree_account_id' => $tree->id,
                'tree_account_title' => $tree->title,
                'tree_account_serial_number' => $tree->serial_number,
                'initial_debit' => $initialDebit,
                'initial_credit' => $initialCredit,
                'initial_balance' => $initialBalance,
                'transactions' => $transactions,
                'period_debit' => $totalDebit,
                'period_credit' => $totalCredit,
                'period_balance' => $totalDebit - $totalCredit,
                'final_balance' => $initialBalance + ($totalDebit - $totalCredit),
                'show_by_default' => $account['currency_id'] == null ? true : false,

            ];
        });
    }

    public function costCenterReport(array $request)
    {
        $reports = CostCenter::whereIn('id', $request['cost_center_ids'] ?? [])
            ->with([
                'transfers' => function ($query) use ($request) {
                    $query->when($request['date_from'] ?? null, function ($query) use ($request) {
                        return $query->where('date', '>=', $request['date_from']);
                    })
                        ->when($request['date_to'] ?? null, function ($query) use ($request) {
                            return $query->where('date', '<=', $request['date_to']);
                        })
                        ->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                            $query->whereHas('constraint', function ($subQuery) use ($request) {
                                $subQuery->whereIn('company_id', $request['company_ids'])
                                    ->where('active', $request['active']);
                            });
                        });
                },
                'transfers.constraint',
                'transfers.treeAccounting',
                'transfers.currency.currentTranslation',
            ])
            ->get();

        return $reports->map(function ($reports) {
            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;

            foreach ($reports->transfers as $transfer) {
                $debit = $transfer->debit ?? 0;
                $credit = $transfer->creditor ?? 0;
                $balance = $debit - $credit;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'tree_account_title' => $transfer->treeAccounting->title,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'debit' => $debit,
                    'credit' => $credit,
                    'balance' => $balance,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'cost_center_id' => $reports->id,
                'cost_center_name' => $reports->name,
                'transactions' => $transactions,
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'total_balance' => $totalDebit - $totalCredit,
            ];
        });
    }

    // كشف خزينة الموظف
    public function getEmployeReports(array $data)
    {

        $targetCurrency = null;
        if (! empty($data['currency_id'])) {
            $targetCurrency = Currency::find($data['currency_id']);
        } else {
            $targetCurrency = Currency::where('is_default', 1)->first();
        }

        if (! $targetCurrency) {
            return false;
        }

        $cashboxMapping = UserAccountMapping::where('user_id', $data['employee_id'])
            ->where('currency_id', $targetCurrency->id)
            ->first();

        if (! $cashboxMapping) {
            $cashboxMapping = UserAccountMapping::where('user_id', $data['employee_id'])
                ->first();
        }

        if (! $cashboxMapping || ! $cashboxMapping->wallet_account_id) {
            return false;
        }

        $cashboxAccountId = $cashboxMapping->wallet_account_id;

        $isBaseCurrency = ($targetCurrency->is_default == 1);

        $debitCol = $isBaseCurrency ? 'debit' : 'currency_debit';
        $creditCol = $isBaseCurrency ? 'creditor' : 'currency_creditor';

        $openingBalance = Transfer::where('tree_accounting_id', $cashboxAccountId)
            ->where('date', '<', $data['from_date'])
            ->when(! $isBaseCurrency, function ($query) use ($targetCurrency) {
                return $query->where('currency_id', $targetCurrency->id);
            })
            ->sum(DB::raw("$debitCol - $creditCol"));

        $transactions = Transfer::with('constraint')
            ->where('tree_accounting_id', $cashboxAccountId)
            ->whereBetween('date', [$data['from_date'], $data['to_date']])
            ->when(! $isBaseCurrency, function ($query) use ($targetCurrency) {
                return $query->where('currency_id', $targetCurrency->id);
            })
            ->orderBy('date', 'asc')
            ->orderBy('id', 'asc')
            ->get();

        $totalDebit = $transactions->sum($debitCol);
        $totalCreditor = $transactions->sum($creditCol);
        $closingBalance = $openingBalance + $totalDebit - $totalCreditor;
        $runningBalance = $openingBalance;
        $processedTransactions = $transactions->map(function ($tx) use (&$runningBalance, $debitCol, $creditCol) {

            $debit = $tx->{$debitCol} ?? 0;
            $creditor = $tx->{$creditCol} ?? 0;
            $runningBalance += ($debit - $creditor);

            return [
                'date' => $tx->date,
                'constraint_name' => $tx->constraint->name ?? $tx->name ?? $tx->id,
                'description' => $tx->note ?? $tx->titles ?? 'N/A',
                'debit' => (float) $debit,
                'creditor' => (float) $creditor,
                'balance' => (float) $runningBalance,
                'transfer_rate' => (float) $tx->currency_transfer_rate,
            ];
        });

        return [
            'totals' => [
                'opening_balance' => (float) $openingBalance,
                'total_debit' => (float) $totalDebit,
                'total_creditor' => (float) $totalCreditor,
                'closing_balance' => (float) $closingBalance,
            ],
            'transactions' => $processedTransactions,
        ];
    }

    // تقرير دفعات الموظفين

    public function EmployeePaymentVouchers(array $data)
    {
        if (empty($data['company_id']) || empty($data['from_date']) || empty($data['to_date'])) {
            return $this->returnEmptyReport($data, 'بيانات الفلترة (الشركة، التواريخ) غير مكتملة.');
        }

        // 1. تحديد أعمدة العملة المستخدمة
        $useCurrency = ! empty($data['currency_id']);
        $colDebit = $useCurrency ? 'currency_debit' : 'debit';
        $colCredit = $useCurrency ? 'currency_creditor' : 'creditor';

        // 2. جلب الفواتير (للحصول على الرقم التسلسلي - serial_num)
        $invoicesQuery = Invoice::query()
            ->where('company_id', $data['company_id'])
            ->whereBetween('invoice_date', [$data['from_date'], $data['to_date']]);

        if (! empty($data['employee_id'])) {
            $invoicesQuery->where('sales_id', $data['employee_id']);
        }

        $invoices = $invoicesQuery
            ->select('id', 'serial_num')
            ->get()
            ->keyBy('id');

        $invoiceIds = $invoices->pluck('id');

        if ($invoiceIds->isEmpty()) {
            return $this->returnEmptyReport($data, 'لا توجد فواتير في هذه الفترة.');
        }

        // 3. جلب حسابات العملاء الآجلة
        $accountsQuery = CompanyAccountMappings::where('company_id', $data['company_id'])
            ->where('type', 'customer_credit');

        $accountIds = $accountsQuery->pluck('tree_account_id')->unique();

        if ($accountIds->isEmpty()) {
            return $this->returnEmptyReport($data, 'لم يتم العثور على حسابات تحصيلات (اجل عملاء).');
        }

        // 4. جلب التحويلات (حركات السداد الفعلية فقط)
        $journalTransfers = Transfer::query()
            ->whereIn('invoice_id', $invoiceIds)
            ->whereIn('tree_accounting_id', $accountIds)
            ->where('capture_exchange', 'payment')
            ->whereBetween('date', [$data['from_date'], $data['to_date']])
            ->whereHas('constraint', function ($q) {
                $q->where('type_optional', 'invoice');
            })
            ->with(['constraint:id,pay_tax,employee_tax', 'payType:id,type'])
            ->select(
                'id', 'constraint_id', 'invoice_id', 'pay_type_id', 'date',
                'debit', 'creditor', 'currency_debit', 'currency_creditor'
            )
            ->orderBy('date', 'asc')
            ->get();

        // 5. بناء التقرير المسطح (Flat Journal) + التجميع
        $reportJournal = [];
        $summaryPaymentMethods = [];
        $summaryTotalPaid = 0;
        $summaryTotalPayCommission = 0;
        $summaryTotalEmployeeCommission = 0;

        foreach ($journalTransfers as $transfer) {

            $invoice = $invoices->get($transfer->invoice_id);
            $invoiceSerial = optional($invoice)->serial_num ?? 'غير معروف';

            // حساب المبلغ المدفوع (Credit - Debit على حساب العميل)
            $valDebit = (float) $transfer->{$colDebit};
            $valCredit = (float) $transfer->{$colCredit};
            $paidAmount = $valCredit - $valDebit;

            // العمولة
            $payCommission = optional($transfer->constraint)->pay_tax ?? 0;
            $employeeCommission = optional($transfer->constraint)->employee_tax ?? 0;

            // --- منطق التجميع (Aggregation Logic) ---
            $summaryTotalPaid += $paidAmount;
            $summaryTotalPayCommission += $payCommission;
            $summaryTotalEmployeeCommission += $employeeCommission;

            $paymentMethod = optional($transfer->payType)->type ?? 'غير محدد';
            if (! isset($summaryPaymentMethods[$paymentMethod])) {
                $summaryPaymentMethods[$paymentMethod] = 0;
            }
            $summaryPaymentMethods[$paymentMethod] += $paidAmount;
            // --- نهاية منطق التجميع ---

            $reportJournal[] = [
                'invoice_number' => $invoiceSerial,
                'amount_paid' => number_format($paidAmount, 2),
                'payment_date' => $transfer->date,
                'constraint_number' => $transfer->constraint_id,
                'payment_method' => $paymentMethod,
                'pay_commission' => number_format($payCommission, 2),
                'employee_commission' => number_format($employeeCommission, 2),
            ];
        }

        // 6. إرجاع النتيجة مع الملخصات الجديدة
        return [
            'status' => 200,
            'message' => 'success',
            'data' => [
                'report_title' => 'تقرير دفعات الموظف (دفتر يومية التحصيل)',
                'filters' => $data,

                // الملخص الأول: ملخص طرق الدفع (لتطابق الصورة العلوية)
                'payment_methods_summary' => $summaryPaymentMethods,

                // الملخص الثاني: ملخص حركة الفواتير (لتطابق الصورة السفلية)
                'invoice_movement_summary' => [
                    'total_paid_from_invoices' => number_format($summaryTotalPaid, 2), // المبالغ المدفوعة من الفواتير
                    'total_payment_commissions' => number_format($summaryTotalPayCommission, 2), // عمولات الدفع
                    'total_employee_commissions' => number_format($summaryTotalEmployeeCommission, 2), // عمولات الموظف
                ],

                // تفاصيل الحركات
                'report_data' => $reportJournal,
            ],
        ];
    }

    private function getTaxAndRevenueSummary(array $invoiceIds, array $data)
    {
        // 1) تحديد الأعمدة حسب العملة
        $useCurrency = ! empty($data['currency_id']);
        $colDebit = $useCurrency ? 'currency_debit' : 'debit';
        $colCredit = $useCurrency ? 'currency_creditor' : 'creditor';

        // 2) حسابات الإيراد والعميل
        $revenueAccountIds = CompanyAccountMappings::where('company_id', $data['company_id'])
            ->where('type', 'package_service_revenue')
            ->pluck('tree_account_id');

        $creditAccountIds = CompanyAccountMappings::where('company_id', $data['company_id'])
            ->where('type', 'customer_credit')
            ->pluck('tree_account_id');

        // 3) الحصول على كل أطراف قيد start_invoice (مع إضافة invoice_id)
        $startInvoiceTransfers = Transfer::query()
            ->whereIn('invoice_id', $invoiceIds)
            ->whereHas('constraint', function ($q) {
                $q->where('type_optional', 'start_invoice');
            })
            //  مهم جداً: إضافة invoice_id هنا لنتمكن من التجميع
            ->select('invoice_id', 'tree_accounting_id', $colDebit, $colCredit)
            ->get();

        $totalRevenue = 0;
        $totalSalesTax = 0;
        $taxPerInvoice = []; // مصفوفة لحفظ الضريبة لكل فاتورة

        // 4) توزيع الحركات وتجميع الضريبة لكل فاتورة
        foreach ($startInvoiceTransfers as $transfer) {
            $amount = (float) $transfer->{$colCredit} - (float) $transfer->{$colDebit};

            // تهيئة القيمة للفاتورة لو لم تكن موجودة
            if (! isset($taxPerInvoice[$transfer->invoice_id])) {
                $taxPerInvoice[$transfer->invoice_id] = 0;
            }

            if ($revenueAccountIds->contains($transfer->tree_accounting_id)) {
                $totalRevenue += $amount;
            } elseif (! $creditAccountIds->contains($transfer->tree_accounting_id)) {
                // أي حساب غير الإيراد وغير آجل العملاء = ضريبة المبيعات
                $totalSalesTax += $amount;

                // إضافة قيمة الضريبة لهذه الفاتورة تحديداً
                $taxPerInvoice[$transfer->invoice_id] += $amount;
            }
        }

        return [
            'total_revenue' => $totalRevenue,
            'total_sales_tax' => $totalSalesTax,
            'tax_per_invoice' => $taxPerInvoice, // إرجاع التفاصيل
        ];
    }

    public function EmployeeInvoiceReport(array $data)
    {
        // 0) التحقق من البيانات
        if (empty($data['company_id']) || empty($data['from_date']) || empty($data['to_date'])) {
            return $this->returnEmptyReport($data, 'بيانات الفلترة غير مكتملة');
        }

        // 1) الأعمدة
        $useCurrency = ! empty($data['currency_id']);
        $colDebit = $useCurrency ? 'currency_debit' : 'debit';
        $colCredit = $useCurrency ? 'currency_creditor' : 'creditor';

        // 2) الفواتير
        $invoicesQuery = Invoice::query()
            ->where('company_id', $data['company_id'])
            ->whereBetween('invoice_date', [$data['from_date'], $data['to_date']]);

        if (! empty($data['employee_id'])) {
            $invoicesQuery->where('sales_id', $data['employee_id']);
        }

        $invoices = $invoicesQuery
            ->select('id', 'serial_num', 'invoice_date', 'invoice_price', 'sales_id', 'customer_id')
            ->orderBy('invoice_date', 'asc')
            ->get();

        $invoiceIds = $invoices->pluck('id');

        if ($invoiceIds->isEmpty()) {
            return $this->returnEmptyReport($data, 'لا توجد فواتير في هذه الفترة');
        }

        // 2.5) حسابات العملاء
        $accountIds = CompanyAccountMappings::where('company_id', $data['company_id'])
            ->where('type', 'customer_credit')
            ->pluck('tree_account_id')
            ->unique();

        if ($accountIds->isEmpty()) {
            return $this->returnEmptyReport($data, 'لا يوجد حسابات آجل عملاء');
        }

        // 3) جلب تفاصيل الضرائب
        $taxAndRevenue = $this->getTaxAndRevenueSummary($invoiceIds->toArray(), $data);
        $totalSalesTax = $taxAndRevenue['total_sales_tax'];
        $taxPerInvoiceMap = $taxAndRevenue['tax_per_invoice'];

        // 4) استعلام الحركات
        $allTransfers = Transfer::query()
            ->whereIn('invoice_id', $invoiceIds)
            ->whereIn('tree_accounting_id', $accountIds)
            ->whereIn('capture_exchange', ['payment', 'constraint'])
            ->whereBetween('date', [$data['from_date'], $data['to_date']])
            ->whereHas('constraint', function ($q) {
                $q->whereIn('type_optional', ['start_invoice', 'invoice']);
            })
            ->with(['constraint:id,type_optional,pay_tax,employee_tax', 'payType:id,type'])
            ->select(
                'id', 'constraint_id', 'invoice_id', 'pay_type_id', 'tree_accounting_id',
                'date', 'capture_exchange',
                'debit', 'creditor',
                'currency_debit', 'currency_creditor'
            )
            ->orderBy('date', 'asc')
            ->get();

        // 5) التجميع
        $reportData = [];
        $summaryTotalOpenBalance = 0;
        $summaryTotalDeductions = 0; // إجمالي المدفوع
        $summaryPaymentByType = [];
        $summaryCommissions = ['pay' => 0, 'employee' => 0];

        foreach ($invoices as $invoice) {

            $invoiceTransfers = $allTransfers->where('invoice_id', $invoice->id);

            // (A) الرصيد الافتتاحي
            $startTransfers = $invoiceTransfers->filter(function ($t) {
                return $t->constraint
                    && $t->constraint->type_optional == 'start_invoice'
                    && $t->capture_exchange == 'constraint';
            });

            $openBalance = 0;
            foreach ($startTransfers as $t) {
                $openBalance += (float) $t->{$colDebit};
            }

            // (B) السدادات
            $normalTransfers = $invoiceTransfers->filter(function ($t) {
                return $t->constraint
                    && $t->constraint->type_optional == 'invoice'
                    && $t->capture_exchange == 'payment';
            });

            $currentBalance = $openBalance;
            $invoiceTotalPaid = 0;

            // العمولات (للعرض فقط)
            $invoicePayTax = 0;
            $invoiceEmployeeTax = 0;

            $paymentRows = [];

            foreach ($normalTransfers as $transfer) {
                $paidAmount = (float) $transfer->{$colCredit} - (float) $transfer->{$colDebit};

                $currentBalance -= $paidAmount;
                $invoiceTotalPaid += $paidAmount;

                $pType = $transfer->payType->type ?? 'N/A';
                if (! isset($summaryPaymentByType[$pType])) {
                    $summaryPaymentByType[$pType] = 0;
                }
                $summaryPaymentByType[$pType] += $paidAmount;

                $currPayTax = $transfer->constraint->pay_tax ?? 0;
                $currEmpTax = $transfer->constraint->employee_tax ?? 0;

                $invoicePayTax += $currPayTax;
                $invoiceEmployeeTax += $currEmpTax;

                $summaryCommissions['pay'] += $currPayTax;
                $summaryCommissions['employee'] += $currEmpTax;

                $paymentRows[] = [
                    'date' => $transfer->date,
                    'constraint_no' => $transfer->constraint_id,
                    'type' => $pType,
                    'amount' => number_format($paidAmount, 2),
                    'balance_after' => number_format($currentBalance, 2),
                ];
            }

            $summaryTotalOpenBalance += $openBalance;
            $summaryTotalDeductions += $invoiceTotalPaid;

            // (C) حساب الربح لهذه الفاتورة (التعديل هنا)
            $thisInvoiceSalesTax = $taxPerInvoiceMap[$invoice->id] ?? 0;

            // حسب طلبك: Net Profit = Final Balance - Sales Tax
            // أي: (المبلغ المتبقي على العميل) - (الضريبة)
            $invoiceNetProfit = $currentBalance - $thisInvoiceSalesTax;

            $reportData[] = [
                'invoice_info' => [
                    'serial' => $invoice->serial_num,
                    'date' => $invoice->invoice_date,
                    'calculated_open_balance' => number_format($openBalance, 2),
                    'currency_mode' => $useCurrency ? 'Foreign Currency' : 'Local Currency',
                ],
                'transactions' => $paymentRows,
                'invoice_summary' => [
                    'total_deducted' => number_format($invoiceTotalPaid, 2),
                    'final_balance' => number_format($currentBalance, 2), // هذا هو الرقم المستخدم في الحساب
                    'sales_tax' => number_format($thisInvoiceSalesTax, 2),
                    'commissions_total' => number_format($invoicePayTax + $invoiceEmployeeTax, 2),
                    'net_profit' => number_format($invoiceNetProfit, 2), // الناتج: المتبقي - الضريبة
                ],
            ];
        }

        // 6) الملخص النهائي

        // إجمالي المتبقي الكلي
        $netRemaining = $summaryTotalOpenBalance - $summaryTotalDeductions;

        // صافي الربح النهائي = إجمالي المتبقي - إجمالي الضرائب
        // ليتطابق مع المنطق المستخدم داخل الفاتورة الواحدة
        $finalNetProfit = $netRemaining - $totalSalesTax;

        $summary = [
            'total_invoices_count' => $invoices->count(),
            'total_open_balances' => number_format($summaryTotalOpenBalance, 2),
            'total_deductions' => number_format($summaryTotalDeductions, 2),
            'net_remaining' => number_format($netRemaining, 2),
            'deductions_by_type' => $summaryPaymentByType,
            'commissions' => [
                'payment_tax' => number_format($summaryCommissions['pay'], 2),
                'employee_tax' => number_format($summaryCommissions['employee'], 2),
            ],
            'total_sales_tax' => number_format($totalSalesTax, 2),
            'final_net_profit' => number_format($finalNetProfit, 2),
        ];

        return [
            'status' => 200,
            'message' => 'success',
            'data' => [
                'filters' => $data,
                'summary' => $summary,
                'details' => $reportData,
            ],
        ];
    }

    private function returnEmptyReport(array $data, string $title = 'تقرير حركة فواتير الموظف')
    {
        return [
            'report_title' => $title,
            'filters' => $data,
            'summary' => [
                'total_invoices' => 0,
                'total_invoice_value' => 0,
                'total_paid_value' => 0,
                'paid_by_type' => [],
                'payment_commission' => 0,
                'employee_commission' => 0,
                'final_profit' => 0,
                'total_remaining' => 0,
            ],
            'report_data' => [],
        ];
    }
}
