<?php

namespace App\sys\Repository\General;

use App\Models\General\Service;
use App\Models\General\ServicesCurrencies;

class ServiceRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'title',
        'type' => 'type',
        'is_fixed_price' => 'is_fixed_price',
        'unit_type' => 'unit_type',
        'category_id' => 'category_id',
        'active' => 'is_active',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $title = request('name', null);
        $limit = request('limit', 15);
        $type = request('type', null);

        return Service::when($title, callback: function ($q, $title) {
            $q->where('title', 'LIKE', "%$title%");
        })->when($type, function ($q, $type) {
            $q->where('type', $type);
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('services_translation', function ($join) {
                    $join->on('services.id', '=', 'services_translation.services_id')
                        ->where('services_translation.lang_id', app('lang_id'));
                })->orderBy('services_translation.title', $order)->select('services.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('servicesCurrencies.currency', 'currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {

        return Service::find($id)->load([
            'currentTranslation',
            'servicesCurrencies.currency',
            'servicesCurrencies.treeAccounting',
            'servicesCurrencies.treeIncomeAccounting',
            'suppliers',
        ]);
    }

    public function create(array $data)
    {
        $service = new Service;

        $service->type = $data['type'];
        $service->is_fixed_price = $data['is_fixed_price'];
        $service->title = $data['name'];
        $service->item_code = $data['item_code'];
        $service->unit_type = $data['unit_type'];
        $service->is_active = $data['active'] ?? 1;

        $service->save();

        foreach ($data['services_currencies'] as $currencyData) {
            $this->createServicesCurrencies($currencyData, $service->id);
        }

        if (isset($data['supplier_ids']) && is_array($data['supplier_ids'])) {
            $service->suppliers()->attach($data['supplier_ids']);
        }
        if (isset($data['user_ids']) && is_array($data['user_ids'])) {

            $service->users()->syncWithPivotValues(
                $data['user_ids'],
                ['type' => $service->type],
                false
            );
        }

        return $service->load([
            'servicesCurrencies.currency',
            'servicesCurrencies.treeAccounting',
            'servicesCurrencies.treeIncomeAccounting',
        ]);
    }

    public function update(Service $service, array $data)
    {
        $service->type = $data['type'] ?? $service->type;
        $service->is_fixed_price = $data['is_fixed_price'] ?? $service->is_fixed_price;
        $service->title = $data['name'] ?? $service->title;
        $service->item_code = $data['item_code'] ?? $service->item_code;
        $service->unit_type = $data['unit_type'] ?? $service->unit_type;
        $service->is_active = $data['active'] ?? $service->is_active;
        $service->save();

        if (isset($data['services_currencies']) && is_array($data['services_currencies'])) {
            foreach ($data['services_currencies'] as $currencyData) {
                $existing = ServicesCurrencies::where('service_id', $service->id)
                    ->where('currency_id', $currencyData['currency_id'])
                    ->first();
                if ($existing) {
                    $existing->tree_accounting_id = $currencyData['tree_accounting_id'] ?? $existing->tree_accounting_id;
                    $existing->tree_income_accounting_id = $currencyData['tree_income_accounting_id'] ?? $existing->tree_income_accounting_id;
                    $existing->adult_price = $currencyData['adult_price'] ?? $existing->adult_price;
                    $existing->child_price = $currencyData['child_price'] ?? $existing->child_price;
                    $existing->save();
                } else {
                    $this->createServicesCurrencies($currencyData, $service->id);
                }
            }
        }

        if (isset($data['supplier_ids']) && is_array($data['supplier_ids'])) {
            $service->suppliers()->sync($data['supplier_ids']);
        }
        if (isset($data['user_ids']) && is_array($data['user_ids'])) {

            $service->users()->syncWithPivotValues(
                $data['user_ids'],
                ['type' => $service->type],
            );
        }

        return $service->load([
            'servicesCurrencies.currency',
            'servicesCurrencies.treeAccounting',
            'servicesCurrencies.treeIncomeAccounting',
        ]);
    }

    public function createServicesCurrencies($data, $serviceId)
    {
        $servicesCurrencies = new ServicesCurrencies;

        $servicesCurrencies->service_id = $serviceId;

        $servicesCurrencies->currency_id = $data['currency_id'];
        $servicesCurrencies->tree_accounting_id = $data['tree_accounting_id'];
        $servicesCurrencies->tree_income_accounting_id = $data['tree_income_accounting_id'];
        $servicesCurrencies->adult_price = $data['adult_price'];
        $servicesCurrencies->child_price = $data['child_price'];

        $servicesCurrencies->save();

        return $servicesCurrencies;
    }

    public function getActive()
    {
        return Service::with([
            'currentTranslation',
        ])->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Service::with('translations')->find($id);
    }

    public function getByType(string $type)
    {
        return Service::with(['currentTranslation', 'servicesCurrencies.currency'])
            ->where('type', $type)
            ->where('is_active', 1)
            ->get();
    }

    public function del(array $ids)
    {
        return Service::whereIn('id', $ids)->delete();
    }

    public function getPricesByServiceId($serviceId, $currencyId)
    {
        return ServicesCurrencies::where('service_id', $serviceId)->where('currency_id', $currencyId)->select(['adult_price', 'child_price'])->get();
    }
}
