<?php

namespace App\sys\Repository\Profile;

use App\Models\Accounting\UnpostedCollections;
use App\Models\Profile\Profile;
use App\Models\Profile\ProfileTraveler;

class CustomerProfilesRepository
{
    public function getCustomerProfiles($customerId, $arrivalDateFrom = null, $arrivalDateTo = null, $departureDateFrom = null, $departureDateTo = null)
    {
        // Define filter variables first
        $arrivalDateFrom = $arrivalDateFrom;
        $departureDateTo = $departureDateTo;

        $query = Profile::with([
            'customer',
            'travelers.currency.currentTranslation',
            'company.currentTranslation',
        ])->where('customer_id', $customerId);

        // Apply date filters using when
        $query->when($arrivalDateFrom, function ($q) use ($arrivalDateFrom) {
            return $q->whereDate('arrival_date', '>=', $arrivalDateFrom);
        });
        $query->when($departureDateTo, function ($q) use ($departureDateTo) {
            return $q->whereDate('departure_date', '<=', $departureDateTo);
        });

        return $query->orderBy('created_at', 'desc')->get();
    }

    public function getProfileCostsByCurrency($profileId)
    {
        return ProfileTraveler::with(['currency.currentTranslation'])
            ->where('profile_id', $profileId)
            ->selectRaw('currency_id, SUM(total_price) as total_cost')
            ->groupBy('currency_id')
            ->get()
            ->map(function ($item) {
                return $this->formatCurrencyData($item, 'total_cost');
            });
    }

    public function getProfilePaymentsByCurrency($profileId)
    {
        return UnpostedCollections::with(['currency.currentTranslation'])
            ->where('profile_id', $profileId)
            ->selectRaw('currency_id, SUM(amount) as total_paid')
            ->groupBy('currency_id')
            ->get()
            ->map(function ($item) {
                return $this->formatCurrencyData($item, 'total_paid');
            });
    }

    /**
     * Format currency data consistently
     */
    private function formatCurrencyData($item, $amountField)
    {

        return [
            'currency_id' => $item->currency_id,
            'currency_name' => $item->currency ? ($item->currency->currentTranslation->name ?? $item->currency->name ?? null) : null,
            'currency_code' => $item->currency ? $item->currency->code : null,
            'currency_symbol' => $item->currency ? $item->currency->symbol : null,
            'exchange_rate' => $item->currency ? $item->currency->exchange_rate : 1,
            'is_default' => $item->currency ? ($item->currency->is_default == 1 ? true : false) : false,
            $amountField => (float) $item->{$amountField},
        ];
    }

    public function getCustomerSummary($customerId, $arrivalDateFrom = null, $arrivalDateTo = null, $departureDateFrom = null, $departureDateTo = null)
    {
        $profiles = $this->getCustomerProfiles($customerId, $arrivalDateFrom, $arrivalDateTo, $departureDateFrom, $departureDateTo);

        return $profiles->map(function ($profile) {
            return $this->buildProfileSummary($profile);
        })->toArray();
    }

    /**
     * Build profile summary with costs and payments
     */
    private function buildProfileSummary($profile)
    {
        $costs = $this->getProfileCostsByCurrency($profile->id);
        $payments = $this->getProfilePaymentsByCurrency($profile->id);

        // Merge costs and payments by currency
        $currencies = collect($costs)->merge($payments)->groupBy('currency_id');

        return [
            'profile_id' => $profile->id,
            'profile_number' => $profile->profile_number,
            'status' => $profile->status,
            'link_code' => $profile->link_code,
            'customer_name' => $profile->customer->full_name ?? null,
            'arrival_date' => $profile->arrival_date,
            'departure_date' => $profile->departure_date,
            'company_name' => $profile->company->currentTranslation->name ?? $profile->company->name_company ?? null,
            'currencies' => $this->mergeCurrencyData($currencies),
        ];
    }

    /**
     * Merge currency costs and payments data
     */
    private function mergeCurrencyData($currencies)
    {
        return $currencies->map(function ($currencyGroup) {
            $cost = $currencyGroup->where('total_cost', '>', 0)->first();
            $payment = $currencyGroup->where('total_paid', '>', 0)->first();

            return [
                'currency_id' => $currencyGroup->first()['currency_id'],
                'currency_name' => $currencyGroup->first()['currency_name'],
                'currency_code' => $currencyGroup->first()['currency_code'],
                'currency_symbol' => $currencyGroup->first()['currency_symbol'],
                'exchange_rate' => $currencyGroup->first()['exchange_rate'],
                'is_default' => $currencyGroup->first()['is_default'],
                'total_cost' => $cost['total_cost'] ?? 0,
                'total_paid' => $payment['total_paid'] ?? 0,
                'remaining' => ($cost['total_cost'] ?? 0) - ($payment['total_paid'] ?? 0),
            ];
        })->values()->toArray();
    }

    /**
     * Get customer financial summary by currency (aggregated across all profiles)
     */
    public function getCustomerFinancialSummary($customerId, $arrivalDateFrom = null, $arrivalDateTo = null, $departureDateFrom = null, $departureDateTo = null)
    {
        // Define filter variables first
        $arrivalDateFrom = $arrivalDateFrom;
        $arrivalDateTo = $arrivalDateTo;
        $departureDateFrom = $departureDateFrom;
        $departureDateTo = $departureDateTo;

        // Get all customer profiles with date filters
        $query = Profile::where('customer_id', $customerId);

        $query->when($arrivalDateFrom, function ($q) use ($arrivalDateFrom) {
            return $q->whereDate('arrival_date', '>=', $arrivalDateFrom);
        });

        $query->when($departureDateTo, function ($q) use ($departureDateTo) {
            return $q->whereDate('departure_date', '<=', $departureDateTo);
        });

        $profileIds = $query->pluck('id')->toArray();

        if (empty($profileIds)) {
            return [];
        }

        // Get total costs by currency from travelers
        $costs = ProfileTraveler::with(['currency.currentTranslation'])
            ->whereIn('profile_id', $profileIds)
            ->selectRaw('currency_id, SUM(total_price) as total_cost')
            ->groupBy('currency_id')
            ->get()
            ->map(function ($item) {
                return $this->formatCurrencyData($item, 'total_cost');
            });

        // Get total payments by currency from unposted collections
        $payments = UnpostedCollections::with(['currency.currentTranslation'])
            ->whereIn('profile_id', $profileIds)
            ->selectRaw('currency_id, SUM(amount) as total_paid')
            ->groupBy('currency_id')
            ->get()
            ->map(function ($item) {
                return $this->formatCurrencyData($item, 'total_paid');
            });

        // Merge costs and payments by currency
        $currencies = collect($costs)->merge($payments)->groupBy('currency_id');

        return $currencies->map(function ($currencyGroup) {
            $cost = $currencyGroup->where('total_cost', '>', 0)->first();
            $payment = $currencyGroup->where('total_paid', '>', 0)->first();

            return [
                'currency_id' => $currencyGroup->first()['currency_id'],
                'currency_name' => $currencyGroup->first()['currency_name'],
                'currency_code' => $currencyGroup->first()['currency_code'],
                'currency_symbol' => $currencyGroup->first()['currency_symbol'],
                'exchange_rate' => $currencyGroup->first()['exchange_rate'],
                'is_default' => $currencyGroup->first()['is_default'],
                'total_cost' => $cost['total_cost'] ?? 0,
                'total_paid' => $payment['total_paid'] ?? 0,
                'remaining' => ($cost['total_cost'] ?? 0) - ($payment['total_paid'] ?? 0),
            ];
        })->values()->toArray();
    }

    /**
     * Account statement: resolve tree accounts from mappings then fetch transfers with constraints
     */
    public function getCustomerAccountStatement(?int $companyId, ?int $currencyId, ?string $dateFrom, ?string $dateTo, $customerId, $typeOptional = null)
    {
        $mappingQuery = \App\Models\General\CompanyAccountMappings::query()
            ->where('type', 'customer_credit')
            ->when($companyId, function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->when($currencyId, function ($q) use ($currencyId) {
                $q->where('currency_id', $currencyId);
            });

        $treeAccountIds = $mappingQuery->pluck('tree_account_id')->unique()->values()->all();
        if (empty($treeAccountIds)) {
            // Signal to service layer that no mapping exists
            return null;
        }

        $transfers = \App\Models\Accounting\Transfer::with(['constraint', 'customer',  'treeAccounting.currentTranslation',
            'currency.currentTranslation'])
            ->whereIn('tree_accounting_id', $treeAccountIds)
            ->whereNotNull('customer_id')
            ->when($typeOptional, function ($q) use ($typeOptional) {
                $q->where('type_optional', $typeOptional);
            })
            ->when($dateFrom, function ($q) use ($dateFrom) {
                $q->where('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q) use ($dateTo) {
                $q->where('date', '<=', $dateTo);
            })->when($customerId, function ($q) use ($customerId) {
                $q->where('customer_id', $customerId);
            })->orderBy('date', 'asc')
            ->orderBy('id', 'asc')
            ->get();

        $runningBalance = 0.0;
        $totalDebit = 0.0;
        $totalCredit = 0.0;

        $rows = $transfers->map(function ($t) use (&$runningBalance, &$totalDebit, &$totalCredit) {
            $debit = (float) ($t->debit ?? 0);
            $credit = (float) ($t->creditor ?? 0);
            $totalDebit += $debit;
            $totalCredit += $credit;
            $runningBalance += ($debit - $credit);

            $constraint = $t->relationLoaded('constraint') ? $t->constraint : null;
            $constraintDate = $constraint->date ?? $t->date;
            $constraintDoc = $constraint->number_doc ?? null;
            $constraintName = $constraint->name ?? null;
            $type = $constraint->capture_exchange ?? null;

            return [
                'date' => $constraintDate,
                'number_doc' => $constraintDoc,
                'customer_name' => $t->customer->full_name ?? null,
                'customer_id' => $t->customer_id ?? null,
                'constraint_name' => $constraintName,
                'description' => $constraint->description,
                'name' => $t->name ?? null,
                'voucher_id' => $constraint->id ?? null,
                'account_name' => $t->treeAccounting && $t->treeAccounting->currentTranslation ? $t->treeAccounting->currentTranslation->title : ($t->treeAccounting->title ?? null),
                'account_num' => $t->treeAccounting->serial_number_dight,
                'type' => $type,
                'debit' => $debit,
                'credit' => $credit,
                'running_balance' => round($runningBalance, 2),
                'transfer_id' => $t->id,
                'tree_accounting_id' => $t->tree_accounting_id,
                'constraint_id' => $t->constraint_id,
            ];
        })->values()->all();

        return [
            'rows' => $rows,
            'totals' => [
                'total_debit' => round($totalDebit, 2),
                'total_credit' => round($totalCredit, 2),
                'difference' => round($totalDebit - $totalCredit, 2),
            ],
        ];
    }

    /**
     * Customer invoice balances grouped by customer (per company or all)
     */
    public function getCustomerInvoiceBalances(?int $companyId)
    {
        // Resolve tree accounts (all currencies) for company (or all)
        $treeAccountIds = \App\Models\General\CompanyAccountMappings::query()
            ->where('type', 'customer_credit')
            ->when($companyId, function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->pluck('tree_account_id')->unique()->values()->all();

        if (empty($treeAccountIds)) {
            return null;
        }

        // Fetch transfers and group by customer_id
        $transfers = \App\Models\Accounting\Transfer::with(['customer'])
            ->whereIn('tree_accounting_id', $treeAccountIds)
            ->whereNotNull('customer_id')
            ->get()
            ->groupBy('customer_id');

        $rows = [];
        $grandDebit = 0.0;
        $grandCredit = 0.0;

        foreach ($transfers as $customerId => $items) {
            $debit = (float) $items->sum(function ($t) {
                return (float) ($t->debit ?? 0);
            });
            $credit = (float) $items->sum(function ($t) {
                return (float) ($t->creditor ?? 0);
            });
            $diff = $debit - $credit;
            $grandDebit += $debit;
            $grandCredit += $credit;

            $first = $items->first();
            $rows[] = [
                'customer_id' => $customerId,
                'customer_name' => $first && $first->customer ? $first->customer->full_name : null,
                'customer_phone' => $first && $first->customer ? ($first->customer->phone ?? null) : null,
                'total_debit' => round($debit, 2),
                'total_credit' => round($credit, 2),
                'difference' => round($diff, 2),
            ];
        }

        return [
            'rows' => $rows,
            'totals' => [
                'customers' => count($rows),
                'total_debit' => round($grandDebit, 2),
                'total_credit' => round($grandCredit, 2),
                'difference' => round($grandDebit - $grandCredit, 2),
            ],
        ];
    }

    /**
     * استرجاع معاملات المحفظة Wallet للعميل بالفلاتر
     */
    public function getCustomerWalletTransactions($data)
    {
        $customerId = $data['customer_id'];
        $from = $data['from'];
        $to = $data['to'];
        $currencyId = $data['currency_id'];
        $customer = \App\Models\Profile\Customer::find($customerId);
        $treeAccountIds = \App\Models\General\CompanyAccountMappings::where('type', 'customer_advance')
            ->where('company_id', $customer->company_id)
            ->pluck('tree_account_id')
            ->unique()
            ->values()
            ->all();
        $query = \App\Models\Accounting\Transfer::with([
            'treeAccounting.currentTranslation',
            'currency.currentTranslation',
            'constraint',
            'customer'])
            ->where('type_optional', 'wallet')
            ->where('customer_id', $customerId)
            ->whereIn('tree_accounting_id', $treeAccountIds)
            ->when($from, function ($q) use ($from) {
                $q->where('date', '>=', $from);
            })->when($to, function ($q) use ($to) {
                $q->where('date', '<=', $to);
            })->when($currencyId, function ($q) use ($currencyId) {
                $q->where('currency_id', $currencyId);
            });
        $transfers = $query->orderBy('date', 'asc')->orderBy('id', 'asc')->get();

        $runningBalance = 0;
        $results = [];
        foreach ($transfers as $item) {
            $debit = (float) ($item->debit ?? 0);
            $creditor = (float) ($item->creditor ?? 0);
            $runningBalance += $debit - $creditor;
            $results[] = [
                'id' => $item->id,
                'date' => $item->date,
                'account_name' => $item->treeAccounting && $item->treeAccounting->currentTranslation ? $item->treeAccounting->currentTranslation->title : ($item->treeAccounting->title ?? null),
                'account_id' => $item->tree_accounting_id,
                'account_num' => $item->treeAccounting->serial_number_dight,
                'debit' => $debit,
                'credit' => $creditor,
                'capture_exchange' => $item->capture_exchange,
                'type_optional' => $item->type_optional,
                'running_balance' => round($runningBalance, 2),
                'currency_id' => $item->currency_id,
                'currency_name' => $item->currency && $item->currency->currentTranslation ? $item->currency->currentTranslation->name : ($item->currency->name ?? null),
                'currency_code' => $item->currency ? $item->currency->code ?? null : null,
                'currency_symbol' => $item->currency ? $item->currency->symbol ?? null : null,
                'constraint_number' => $item->constraint ? $item->constraint->number_doc : null,
                'voucher_id' => $item->constraint_id,
                'name' => $item->name,
                'description' => $item->constraint->description,
                'customer_id' => $item->customer_id,
                'customer_name' => $item->customer ? ($item->customer->full_name ?? null) : null,
            ];
        }

        return $results;
    }

    /**
     * إجمالي رصيد المحفظة Wallet فقط (debit-credit)
     */
    public function getCustomerWalletBalance($data)
    {

        $customerId = $data['customer_id'];

        // جلب company_id من العميل
        $customer = \App\Models\Profile\Customer::find($customerId);
        if (! $customer || ! $customer->company_id) {
            return [
                'balance' => 0.0,
                'total_debit' => 0.0,
                'total_credit' => 0.0,
            ];
        }

        // جلب tree_account_id من company_account_mappings
        // حيث type = 'customer_advance' و company_id = company_id من العميل
        $treeAccountIds = \App\Models\General\CompanyAccountMappings::where('type', 'customer_advance')
            ->where('company_id', $customer->company_id)
            ->pluck('tree_account_id')
            ->unique()
            ->values()
            ->all();

        if (empty($treeAccountIds)) {
            return [
                'balance' => 0.0,
                'total_debit' => 0.0,
                'total_credit' => 0.0,
            ];
        }

        // جلب Transfers بناءً على tree_account_id و customer_id
        $query = \App\Models\Accounting\Transfer::where('type_optional', 'wallet')
            ->where('customer_id', $customerId)
            ->whereIn('tree_accounting_id', $treeAccountIds);

        $debit = (float) $query->sum('debit');
        $creditor = (float) $query->sum('creditor');
        $balance = $debit - $creditor;

        return [
            'balance' => round($balance, 2),
            'total_debit' => round($debit, 2),
            'total_credit' => round($creditor, 2),
        ];
    }

    /**
     * ملخص محافظ العملاء: اسم العميل، إجمالي مدين، إجمالي دائن، والفرق
     * filters: customer_ids (array|null), from (date|null), to (date|null)
     */
    public function getCustomersWalletSummary(array $data)
    {
        $customerIds = $data['customer_ids'] ?? null; // array|null
        $from = $data['from'] ?? null;
        $to = $data['to'] ?? null;

        // جلب العملاء المطلوبين
        $customersQuery = \App\Models\Profile\Customer::query();
        if (is_array($customerIds) && count($customerIds) > 0) {
            $customersQuery->whereIn('id', $customerIds);
        }
        $customers = $customersQuery->get(['id', 'full_name', 'company_id']);
        if ($customers->isEmpty()) {
            return [];
        }

        // جلب جميع المابات للحسابات customer_advance لمجموعة الشركات المعنية دفعة واحدة
        $companyIds = $customers->pluck('company_id')->filter()->unique()->values()->all();
        $mappings = \App\Models\General\CompanyAccountMappings::where('type', 'customer_advance')
            ->whereIn('company_id', $companyIds ?: [0])
            ->get()
            ->groupBy('company_id');

        $results = [];
        $noSpecificCustomers = ! is_array($customerIds) || count($customerIds) === 0;
        foreach ($customers as $customer) {
            $companyMappings = $mappings->get($customer->company_id);
            $treeAccountIds = $companyMappings
                ? $companyMappings->pluck('tree_account_id')->unique()->values()->all()
                : [];

            if (empty($treeAccountIds)) {
                if ($noSpecificCustomers) {
                    // تجاهل العملاء الذين لا يمتلكون مابات عندما لا يتم تحديد عملاء معينين
                    continue;
                }
                // إن تم تحديد عملاء، أعِد صفراً لعدم وجود مابات
                $results[] = [
                    'customer_id' => $customer->id,
                    'customer_name' => $customer->full_name,
                    'total_debit' => 0.0,
                    'total_credit' => 0.0,
                    'difference' => 0.0,
                ];

                continue;
            }

            $q = \App\Models\Accounting\Transfer::where('type_optional', 'wallet')
                ->where('customer_id', $customer->id)
                ->whereIn('tree_accounting_id', $treeAccountIds)
                ->when($from, function ($qq) use ($from) {
                    $qq->where('date', '>=', $from);
                })
                ->when($to, function ($qq) use ($to) {
                    $qq->where('date', '<=', $to);
                });

            $totalDebit = (float) $q->sum('debit');
            $totalCredit = (float) $q->sum('creditor');
            $diff = $totalDebit - $totalCredit;

            // لو المستخدم لم يحدد عملاء بعينهم، تجاهل النتائج ذات صفر مدين وصفر دائن
            if ($noSpecificCustomers && ($totalDebit == 0.0 && $totalCredit == 0.0)) {
                continue;
            }

            $results[] = [
                'customer_id' => $customer->id,
                'customer_name' => $customer->full_name,
                'total_debit' => round($totalDebit, 2),
                'total_credit' => round($totalCredit, 2),
                'difference' => round($diff, 2),
            ];
        }

        return $results;
    }
}
