<?php

namespace App\sys\Services\Accounting;

use App\Models\Accounting\Constraint;
use App\sys\Helper;
use App\sys\Repository\Accounting\ConstraintRepository;
use App\sys\Repository\Profile\AttachmentRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class ConstraintService extends Services
{
    protected $capture_exchange_types = ['receipt', 'payment'];

    protected ConstraintRepository $constraintRepository;

    protected AttachmentRepository $attachmentRepository;

    public function __construct(ConstraintRepository $constraintRepository, AttachmentRepository $attachmentRepository)
    {
        $this->constraintRepository = $constraintRepository;
        $this->attachmentRepository = $attachmentRepository;
    }

    public function getById(int $id)
    {
        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->constraintRepository->findById($id);
    }

    public function getConstriantByid($constraint_id)
    {
        $validator = Validator::make(['id' => $constraint_id], [
            'id' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $data = $this->constraintRepository->getConstriantByid($constraint_id);

        return [
            'success' => true,
            'data' => $data,
        ];

    }

    public function print($constraint_id)
    {
        $validator = Validator::make(['id' => $constraint_id], [
            'id' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->constraintRepository->print($constraint_id);
    }

    public function printDailyConstraint($constraint_id)
    {
        $validator = Validator::make(['id' => $constraint_id], [
            'id' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->constraintRepository->printDailyConstraint($constraint_id);
    }

    public function create(array $request)
    {
        $rules = [
            'profile_id' => ['nullable', 'exists:pr_profile,id'],
            'type_of_movement_id' => ['nullable', 'integer'],
            'capture_exchange' => ['required', 'string', 'in:constraint,receipt,payment'],
            'number_movement' => ['nullable', 'integer'],
            'date' => ['required', 'date'],
            'name' => ['nullable', 'string', 'max:200'],
            'description' => ['nullable', 'string'],
            'difference' => ['nullable', 'numeric'],
            'active' => ['required', 'boolean'],
            'user_id' => ['nullable', 'integer'],
            'type_of_movement' => ['nullable', 'integer'],
            'number_chek' => ['nullable', 'max:50'],
            'chek_date' => ['nullable', 'date'],
            'username' => ['nullable', 'string', 'max:200'],
            'company_id' => ['required', 'integer', 'exists:companies,id'],
            'customer_id' => ['nullable', 'integer'],
            'supplier_id' => ['nullable', 'integer'],
            'user_id_emp' => ['nullable', 'integer'],
            'invoice_id' => ['nullable', 'integer'],
            'duration' => ['nullable', 'integer'],
            'due_date' => ['nullable', 'date'],
            'worthy_days' => ['nullable', 'integer'],
            'payType_id' => ['nullable', 'integer'],
            'employee_tax' => ['nullable', 'numeric'],
            'pay_tax' => ['nullable', 'numeric'],
            'currency_id' => ['nullable', 'integer', 'exists:currencies,id'],
            'transfers' => ['required', 'array', 'min:1'],
            'transfers.*.name' => ['nullable', 'string', 'max:200'],
            'transfers.*.creditor' => ['required', 'numeric', 'min:0'],
            'transfers.*.debit' => ['required', 'numeric', 'min:0'],
            'transfers.*.tree_accounting_id' => ['required', 'integer', 'exists:tree_accounting,id'],
            'transfers.*.cost_center_id' => ['nullable', 'integer', 'exists:cost_centers,id'],
            'attachment' => ['sometimes', 'array'],
            'attachment.*.name' => ['nullable', 'string', 'max:600'],
            'attachment.*.attachment' => ['nullable', 'string'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        if (in_array($request['capture_exchange'], $this->capture_exchange_types) && count($request['transfers']) !== 2) {
            $this->setError(['For receipt or payment, exactly 2 transfers are required']);

            return false;
        }

        foreach ($request['transfers'] as $index => $transfer) {
            if ($transfer['creditor'] > 0 && $transfer['debit'] > 0) {
                $this->setError(["transfers.$index" => 'Only one of creditor or debit can be non-zero']);

                return false;
            }
            if ($transfer['creditor'] == 0 && $transfer['debit'] == 0) {
                $this->setError(["transfers.$index" => 'Either creditor or debit must be non-zero']);

                return false;
            }
            // السماح باختلاف عملة الحساب عن عملة القيد/الإدخال - سيتم التحويل لاحقاً
        }

        // احسب المجاميع بناءً على العملة
        $incomingCurrencyId = $request['currency_id'] ?? null;
        $defaultCurrencyId = \App\Models\General\Currency::where('is_default', 1)->value('id');
        $incomingRate = 1;
        if ($incomingCurrencyId && $incomingCurrencyId != $defaultCurrencyId) {
            $incomingRate = (float) (\App\Models\General\Currency::find($incomingCurrencyId)->exchange_rate ?? 1);
        }

        // إذا كانت العملة غير افتراضية نضرب في rate لتحويلها إلى الافتراضية
        if ($incomingCurrencyId && $incomingCurrencyId != $defaultCurrencyId) {
            $totalCreditor = 0;
            $totalDebit = 0;
            foreach ($request['transfers'] as $t) {
                $totalCreditor += ((float) $t['creditor']) * $incomingRate;
                $totalDebit += ((float) $t['debit']) * $incomingRate;
            }
        } else {
            // العملة الافتراضية (أو غير محددة) المجاميع كما هي
            $totalCreditor = array_sum(array_column($request['transfers'], 'creditor'));
            $totalDebit = array_sum(array_column($request['transfers'], 'debit'));
        }
        if ($totalCreditor !== $totalDebit) {
            $this->setError(['difference' => 'Total creditor must equal total debit']);

            return false;
        }

        $request['total_creditor'] = $totalCreditor;
        $request['total_debit'] = $totalDebit;

        $constraint = $this->constraintRepository->create($request);

        if ($constraint && isset($request['attachment'])) {
            foreach ($request['attachment'] as $attachment) {
                $attachReq = [
                    'profile_id' => $request['profile_id'] ?? null,
                    'attachable_type' => (new Constraint)->getMorphClass(),
                    'attachable_id' => $constraint->id,
                    'attachment' => $attachment['attachment'],
                    'name' => $attachment['name'],
                ];
                $saved = $this->addAttachment($attachReq);
            }
        }

        return $constraint;
    }

    public function update(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:constraints,id,active,0'],
            'profile_id' => ['sometimes', 'nullable', 'exists:pr_profile,id'],
            'type_of_movement_id' => ['sometimes', 'integer'],
            'number_movement' => ['sometimes', 'integer'],
            'date' => ['sometimes', 'nullable', 'date'],
            'name' => ['sometimes', 'nullable', 'string', 'max:200'],
            'description' => ['sometimes', 'nullable', 'string'],
            'difference' => ['sometimes', 'numeric'],
            'active' => ['sometimes', 'boolean'],
            'user_id' => ['sometimes', 'integer'],
            'type_of_movement' => ['sometimes', 'integer'],
            'number_chek' => ['sometimes', 'nullable', 'max:50'],
            'chek_date' => ['sometimes', 'nullable', 'date'],
            'username' => ['sometimes', 'nullable', 'string', 'max:200'],
            'company_id' => ['sometimes', 'integer', 'exists:companies,id'],
            'customer_id' => ['sometimes', 'nullable', 'integer'],
            'supplier_id' => ['sometimes', 'nullable', 'integer'],
            'user_id_emp' => ['sometimes', 'nullable', 'integer'],
            'invoice_id' => ['sometimes', 'nullable', 'integer'],
            'duration' => ['sometimes', 'nullable', 'integer'],
            'due_date' => ['sometimes', 'nullable', 'date'],
            'worthy_days' => ['sometimes', 'integer'],
            'payType_id' => ['sometimes', 'integer'],
            'employee_tax' => ['sometimes', 'numeric'],
            'pay_tax' => ['sometimes', 'nullable', 'numeric'],
            'currency_id' => ['sometimes', 'nullable', 'integer', 'exists:currencies,id'],
            'transfers' => ['required', 'array', 'min:1'],
            'transfers.*.name' => ['nullable', 'string', 'max:200'],
            'transfers.*.creditor' => ['required', 'numeric', 'min:0'],
            'transfers.*.debit' => ['required', 'numeric', 'min:0'],
            'transfers.*.tree_accounting_id' => ['required', 'integer', 'exists:tree_accounting,id'],
            'transfers.*.cost_center_id' => ['nullable', 'integer', 'exists:cost_centers,id'],
            'transfers.*.currency_transfer_rate' => ['required', 'numeric', 'min:1'],
            'attachment' => ['sometimes', 'nullable', 'array'],
            'attachment.*.name' => ['nullable', 'string', 'max:600'],
            'attachment.*.attachment' => ['nullable', 'string'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        if (isset($request['capture_exchange']) && in_array($request['capture_exchange'], $this->capture_exchange_types) && count($request['transfers']) !== 2) {
            $this->setError(['transfers' => 'For receipt or payment, exactly 2 transfers are required']);

            return false;
        }

        foreach ($request['transfers'] as $index => $transfer) {
            if ($transfer['creditor'] > 0 && $transfer['debit'] > 0) {
                $this->setError(["transfers.$index" => 'Only one of creditor or debit can be non-zero']);

                return false;
            }
            if ($transfer['creditor'] == 0 && $transfer['debit'] == 0) {
                $this->setError(["transfers.$index" => 'Either creditor or debit must be non-zero']);

                return false;
            }
            // السماح باختلاف عملة الحساب عن عملة القيد/الإدخال - سيتم التحويل لاحقاً
        }

        // احسب المجاميع بناءً على العملة
        $incomingCurrencyId = $request['currency_id'] ?? null;
        $defaultCurrencyId = \App\Models\General\Currency::where('is_default', 1)->value('id');
        $incomingRate = 1;
        if ($incomingCurrencyId && $incomingCurrencyId != $defaultCurrencyId) {
            $incomingRate = (float) (\App\Models\General\Currency::find($incomingCurrencyId)->exchange_rate ?? 1);
        }

        // إذا كانت العملة غير افتراضية نضرب في rate لتحويلها إلى الافتراضية
        if ($incomingCurrencyId && $incomingCurrencyId != $defaultCurrencyId) {
            $totalCreditor = 0;
            $totalDebit = 0;
            foreach ($request['transfers'] as $t) {
                $totalCreditor += ((float) $t['creditor']) * $incomingRate;
                $totalDebit += ((float) $t['debit']) * $incomingRate;
            }
        } else {
            // العملة الافتراضية (أو غير محددة) المجاميع كما هي
            $totalCreditor = array_sum(array_column($request['transfers'], 'creditor'));
            $totalDebit = array_sum(array_column($request['transfers'], 'debit'));
        }
        if ($totalCreditor !== $totalDebit) {
            $this->setError(['difference' => 'Total creditor must equal total debit']);

            return false;
        }

        $request['total_creditor'] = $totalCreditor;
        $request['total_debit'] = $totalDebit;

        $constraint = $this->getById($request['id']);
        $updated = $this->constraintRepository->update($constraint, $request);

        if ($updated && isset($request['attachment'])) {
            foreach ($request['attachment'] as $attachment) {
                $attachReq = [
                    'profile_id' => $request['profile_id'] ?? ($updated->profile_id ?? null),
                    'attachable_type' => (new Constraint)->getMorphClass(),
                    'attachable_id' => $updated->id,
                    'name' => $attachment['name'] ?? null,
                    'attachment' => $attachment['attachment'],
                ];
                $this->addAttachment($attachReq);
            }

        }

        return $updated;
    }

    public function del(array $ids)
    {
        $validator = Validator::make(['ids' => $ids], [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->constraintRepository->del($ids);
    }

    public function getPaginated()
    {
        return $this->constraintRepository->getPaginated();
    }

    public function addAttachment(array $request)
    {
        if (isset($request['constraint_id'])) {
            $request['attachable_type'] = (new Constraint)->getMorphClass();
            $request['attachable_id'] = (int) $request['constraint_id'];
        }

        $rules = [
            'profile_id' => ['nullable', 'integer', 'exists:pr_profile,id'],
            'attachable_type' => ['required', 'string'],
            'attachable_id' => ['required', 'integer'],
            'attachment' => ['required', 'string'],
        ];

        if (! empty($request['attachable_type']) && class_exists($request['attachable_type'])) {
            $className = $request['attachable_type'];
            $request['attachable_type'] = (new $className)->getMorphClass();
        }

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $saveAttachment = Helper::saveFiles($request['attachment'], 'upload/constraint');
        if (! $saveAttachment['status']) {
            $this->setError(['attachment' => $saveAttachment['errors']]);

            return false;
        }
        $request['path'] = $saveAttachment['path'];

        return $this->attachmentRepository->upload($request);
    }

    public function deleteAttachments(array $request)
    {
        $validator = Validator::make($request, [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_attachments,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $attachments = \App\Models\Profile\Attachment::whereIn('id', $request['ids'])->get();
        foreach ($attachments as $attachment) {
            $fullPath = public_path($attachment->path);
            if (file_exists($fullPath)) {
                @unlink($fullPath);
            }
        }

        return $this->attachmentRepository->deleteByIds($request['ids']);
    }

    public function getNextDocumentNumber($type)
    {
        return $this->constraintRepository->getNextDocumentNumber($type);
    }

    public function activate(int $id)
    {
        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:constraints,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->constraintRepository->activateById($id);
    }
}
