<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\ReportRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class ReportService extends Services
{
    protected ReportRepository $reportRepository;

    public function __construct(ReportRepository $reportRepository)
    {
        $this->reportRepository = $reportRepository;
    }

    public function getReports()
    {
        return $this->reportRepository->getReports();
    }

    public function getHierarchicalTreeAccountReport()
    {
        $validator = Validator::make(request()->all(), [
            'currency_id' => ['sometimes', 'integer', 'exists:currencies,id'],
            'date_from' => ['sometimes', 'nullable', 'date'],
            'date_to' => ['sometimes', 'date'],
            'type' => ['required', 'in:balance,income'], // التغير في income
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);
        $type = request('type', 'balance');

        return $this->reportRepository->getHierarchicalTreeAccountReport($currency, $companyIds, $dateFrom, $dateTo, $type);
    }

    public function getHierarchicalTreeAccountTrailBalance()
    {
        $validator = Validator::make(request()->all(), [
            'currency_id' => ['sometimes', 'integer', 'exists:currencies,id'],
            'date_from' => ['nullable', 'date'],
            'date_to' => ['required', 'date'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->getHierarchicalTreeAccountReportTrailBalance();
    }

    public function treeAccountTransfers(array $request)
    {

        $validator = Validator::make($request, [
            'treeAccount_ids' => ['required', 'array', 'min:1'],
            'treeAccount_ids.*' => ['required', 'integer', 'exists:tree_accounting,id'],
            'show_by_default' => ['required', 'boolean'],
            'date_from' => ['nullable', 'date', 'date_format:Y-m-d'],
            'date_to' => ['nullable', 'date', 'date_format:Y-m-d', 'after_or_equal:date_from'],
            'company_ids' => ['nullable', 'array'],
            'company_ids.*' => ['integer', 'exists:companies,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->treeAccountTransfer($request);
    }

    public function treeAccountTransfersWithInitialBalance(array $request)
    {
        $validator = Validator::make($request, [
            'treeAccount_ids' => ['required', 'array', 'min:1'],
            'treeAccount_ids.*' => ['required', 'exists:tree_accounting,id'],
            'show_by_default' => ['required', 'boolean'],
            'date_from' => ['nullable', 'date', 'date_format:Y-m-d'],
            'date_to' => ['nullable', 'date', 'date_format:Y-m-d', 'after_or_equal:date_from'],
            'company_ids' => ['nullable', 'array'],
            'company_ids.*' => ['nullable', 'exists:companies,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->treeAccountTransferWithInitialBalance($request);
    }

    public function costCenterReport(array $request)
    {
        $validator = Validator::make($request, [
            'cost_center_ids' => ['required', 'array', 'min:1'],
            'cost_center_ids.*' => ['required', 'integer', 'exists:cost_centers,id'],
            'date_from' => ['nullable', 'date', 'date_format:Y-m-d'],
            'date_to' => ['nullable', 'date', 'date_format:Y-m-d', 'after_or_equal:date_from'],
            'company_ids' => ['nullable', 'array'],
            'company_ids.*' => ['integer', 'exists:companies,id'],
            'active' => ['required', 'boolean'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->costCenterReport($request);
    }

    public function getEmployeReports($data)
    {

        $validator = Validator::make($data, [
            'employee_id' => 'required|exists:users,id',
            'from_date' => 'required|date_format:Y-m-d',
            'to_date' => 'required|date_format:Y-m-d|after_or_equal:from_date',
            'currency_id' => 'nullable|integer|exists:currencies,id',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->getEmployeReports($data);

    }

    public function EmployeePaymentVouchers($request)
    {

        $validator = Validator::make($request, [
            'employee_id' => 'nullable|integer|exists:users,id',
            'company_id' => 'nullable|integer|exists:companies,id',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->EmployeePaymentVouchers($request);

    }

    public function getEmployeeInvoiceReport(array $data)
    {
        $validator = Validator::make($data, [
            'employee_id' => 'nullable|integer|exists:users,id',
            'company_id' => 'nullable|integer|exists:companies,id',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'from_date' => 'required|date_format:Y-m-d',
            'to_date' => 'required|date_format:Y-m-d|after_or_equal:from_date',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reportRepository->EmployeeInvoiceReport($data);
    }
}
