<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\sys\Enums\ServicesType;

class TourRepsHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        return [
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'supplier_id' => $request['service_provider'] == 'supplier_id' ? ['required', 'integer', 'exists:su_supplier,id'] : ['nullable'],
            'service_provider' => ['required', 'in:supplier_id,guide_id'],
            'guide_id' => $request['service_provider'] == 'guide_id' ? ['required', 'integer', 'exists:users,id'] : ['nullable'],
            'guide_language_id' => ['nullable', 'integer', 'exists:guide_language,id'],
            'people_count' => ['required', 'integer', 'min:0'],
            'daily_cost' => ['required', 'numeric', 'min:0'],
            'tip_amount' => ['required', 'numeric', 'min:0'],
            'extra_cost' => ['nullable', 'numeric', 'min:0'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'daily_program_id' => ['required', 'integer', 'exists:pr_daily_programs,id'],

        ];
    }

    public function getUpdateRules(array $request): array
    {
        $rules = [
            'total_tips' => 'prohibited',
            'grand_total' => 'prohibited',
            'currency_rate' => 'prohibited',
            'is_by_handling' => 'nullabel|boolean',
        ];

        if (array_key_exists('service_id', $request)) {
            $rules['service_id'] = ['required', 'integer', 'exists:services,id'];
        }
        if (array_key_exists('supplier_id', $request)) {
            $rules['supplier_id'] = ['nullable', 'integer', 'exists:su_supplier,id'];
        }
        if (array_key_exists('service_provider', $request)) {
            $rules['service_provider'] = ['required', 'in:supplier_id,guide_id'];
        }
        if (array_key_exists('guide_id', $request)) {
            $rules['guide_id'] = ['nullable', 'integer', 'exists:users,id'];
        }
        if (array_key_exists('guide_language_id', $request)) {
            $rules['guide_language_id'] = ['nullable', 'integer', 'exists:languages,id'];
        }
        if (array_key_exists('people_count', $request)) {
            $rules['people_count'] = ['required', 'integer', 'min:0'];
        }
        if (array_key_exists('daily_cost', $request)) {
            $rules['daily_cost'] = ['required', 'numeric', 'min:0'];
        }
        if (array_key_exists('tip_amount', $request)) {
            $rules['tip_amount'] = ['required', 'numeric', 'min:0'];
        }
        if (array_key_exists('extra_cost', $request)) {
            $rules['extra_cost'] = ['nullable', 'numeric', 'min:0'];
        }
        if (array_key_exists('executive_id', $request)) {
            $rules['executive_id'] = ['nullable', 'integer', 'exists:users,id'];
        }
        if (array_key_exists('currency_id', $request)) {
            $rules['currency_id'] = ['required', 'integer', 'exists:currencies,id'];
        }
        if (array_key_exists('notes', $request)) {
            $rules['notes'] = ['nullable', 'string', 'max:1000'];
        }
        if (array_key_exists('profile_id', $request)) {
            $rules['profile_id'] = ['required', 'integer', 'exists:pr_profile,id'];
        }
        if (array_key_exists('daily_program_id', $request)) {
            $rules['daily_program_id'] = ['nullable', 'integer', 'exists:pr_daily_programs,id'];
        }

        if (array_key_exists('is_by_handling', $request)) {
            $rules['is_by_handling'] = ['nullable', 'boolean'];
        }

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        // Normalize provider: if service_provider specified, nullify the other id
        $provider = $request['service_provider'] ?? null;
        if ($provider == 'supplier_id') {
            $request['guide_id'] = null;
        } elseif ($provider == 'guide_id') {
            $request['supplier_id'] = null;
            // keep supplier required; do not nullify supplier per business rules
        }
        // Validate service type
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
            return ['errors' => ['service_id' => ['service type must be tour_reps']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Set currency rate
        $currency = Currency::find($request['currency_id']);
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Compute totals
        $peopleCount = (int) $request['people_count'];
        $dailyCost = (float) $request['daily_cost'];
        $tipAmount = (float) $request['tip_amount'];
        $extraCost = (float) ($request['extra_cost'] ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        // Normalize provider switching on update
        if (array_key_exists('service_provider', $request)) {
            if ($request['service_provider'] === 'supplier_id') {
                $request['guide_id'] = null;
            }
        } else {
            if (array_key_exists('supplier_id', $request) && ! is_null($request['supplier_id'])) {
                $request['guide_id'] = null;
            }
        }
        // If service_id provided, ensure it's correct type
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
                return ['errors' => ['service_id' => ['service type must be tour_reps']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Refresh currency rate if currency_id provided
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        // Use provided values or fallback to existing for calculations
        $peopleCount = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $tipAmount = (float) ($request['tip_amount'] ?? $existing->tip_amount ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? $existing->extra_cost ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
    }
}
