<?php

namespace App\sys\Services\User;

use App\sys\Repository\User\UserRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class UserService extends Services
{
    protected $userrepo;

    public function __construct(UserRepository $userrepo)
    {
        $this->userrepo = $userrepo;

    }

    public function getPaginated()
    {
        // نستدعي الريبوزيتوري مباشرة
        return $this->userrepo->getPaginated();

    }

    public function get($id)
    {

        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:users,id'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($id);

        return $user;
    }

    public function create(array $data)
    {

        $rules = [
            'name' => 'required|string|max:255',
            'first_name' => 'nullable|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'is_active' => 'boolean',
            'employee_number' => 'nullable|string|max:50',

            'role_id' => 'required|integer|exists:roles,id',
            'is_supplier' => 'sometimes|boolean',

            'groups' => 'nullable|array',
            'groups.*' => 'integer|exists:groups,id',

            'services' => 'nullable|array',
            'services.*' => 'integer|exists:services,id',

            'accommodations' => 'nullable|array',
            'accommodations.*' => 'integer|exists:ac_accommodations,id',

            'companies' => 'nullable|array',
            'companies.*' => 'integer|exists:companies,id',

            'accounting' => 'required|array|min:1',
            'accounting.*.currency_id' => 'required|integer|exists:currencies,id',
            'accounting.*.percentage' => 'required|numeric|min:0|max:100',
            'accounting.*.commission_account_id' => 'required|integer|exists:tree_accounting,id',
            'accounting.*.wallet_account_id' => 'required|integer|exists:tree_accounting,id',
            'accounting.*.Supplier_account_id' => 'sometimes|integer|exists:tree_accounting,id',
        ];

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        try {
            return $this->userrepo->create($data);
        } catch (\Exception $e) {
            Log::error('Failed to create user: '.$e->getMessage());
            $this->setError(['database' => ['An error occurred while creating the user.']]);

            return false;
        }
    }

    public function update($id, $data)
    {
        $rules = [
            'name' => 'sometimes|string|max:255',
            'first_name' => 'sometimes|nullable|string|max:255',
            'last_name' => 'sometimes|nullable|string|max:255',
            'phone' => ['sometimes', 'string', Rule::unique('users')->ignore($id), 'max:50'],
            'email' => ['sometimes', 'email', Rule::unique('users')->ignore($id), 'max:255'],
            'password' => ['sometimes', 'nullable', 'confirmed', Password::min(8)],
            'is_active' => 'sometimes|boolean',
            'employee_number' => 'sometimes|nullable|string|max:100',
            'role_id' => 'sometimes|exists:roles,id',
            'is_supplier' => 'sometimes|boolean',

            'groups' => 'sometimes|array',
            'groups.*' => 'exists:groups,id',

            'services' => 'sometimes|array',
            'services.*' => 'exists:services,id',

            'accommodations' => 'sometimes|array',
            'accommodations.*' => 'exists:ac_accommodations,id',

            'companies' => 'sometimes|array',
            'companies.*' => 'exists:companies,id',

            'accounting' => 'sometimes|array',
            'accounting.*.currency_id' => 'sometimes|required|exists:currencies,id',
            'accounting.*.percentage' => 'sometimes|required|numeric|min:0|max:100',
            'accounting.*.commission_account_id' => 'sometimes|required|exists:tree_accounting,id',
            'accounting.*.wallet_account_id' => 'sometimes|required|exists:tree_accounting,id',
            'accounting.*.Supplier_account_id' => 'sometimes|exists:tree_accounting,id',
        ];

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        if (empty($data['password'])) {
            unset($data['password']);
        }

        return $this->userrepo->update($id, $data);

    }

    public function delete($id)
    {

        return $this->userrepo->delete($id);

    }

    public function updateOverrides(int $userId, array $overrides)
    {

        $permissionIds = array_keys($overrides);
        $validator = Validator::make([
            'user_id' => $userId,
            'permission_ids' => $permissionIds,
            'overrides' => $overrides,
        ], [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'permission_ids.*' => ['integer', 'exists:permissions,id'],
            'overrides.*.override_type' => ['required', Rule::in(['allow', 'deny'])],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($userId);
        if (! $user) {
            $this->setError(['user' => 'User not found']);

            return false;
        }

        $this->userrepo->syncOverrides($userId, $overrides);

        // $user->clearPermissionCache();

        return true;
    }

    public function updateUserRole(int $userId, ?int $roleId)
    {
        // 1. [جديد] - إضافة Validation
        $validator = Validator::make([
            'user_id' => $userId,
            'role_id' => $roleId,
        ], [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'role_id' => ['nullable', 'integer', 'exists:roles,id'], // 'nullable' عشان نسمح بإلغاء الـ Role
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($userId);
        if (! $user) {
            $this->setError(['user' => 'User not found']);

            return false;
        }

        $this->userrepo->setRole($userId, $roleId);

        // $user->clearPermissionCache();

        return true;
    }
}
