<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\PaytypeCollection;
use App\Http\Resources\Accounting\PaytypeHistoryResource;
use App\Http\Resources\Accounting\PaytypeLangResources;
use App\Http\Resources\Accounting\PaytypeResource;
use App\sys\ApiResponse;
use App\sys\Services\Accounting\PaytypeService;
use App\sys\Services\Accounting\PaytypeTranslationService;
use Illuminate\Http\Request;

class PaytypeController extends Controller
{
    use ApiResponse;

    protected PaytypeService $paytypeService;

    protected PaytypeTranslationService $paytypeTranslationService;

    public function __construct(PaytypeService $paytypeService, PaytypeTranslationService $paytypeTranslationService)
    {
        $this->paytypeService = $paytypeService;
        $this->paytypeTranslationService = $paytypeTranslationService;
    }

    public function index()
    {
        $paytypes = $this->paytypeService->getPaginatedPaytypes();

        return $this->apiResponse(200, 'success', null, new PaytypeCollection($paytypes));
    }

    public function getByCurrency($currency)
    {
        $paytypes = $this->paytypeService->getByCurrnecy($currency);

        return $this->apiResponse(200, 'success', null, new PaytypeCollection($paytypes));
    }

    public function show(int $id)
    {
        if ($paytype = $this->paytypeService->getPaytypeById($id)) {
            return $this->apiResponse(200, 'success', null, new PaytypeResource($paytype));
        }
        $errors = $this->paytypeService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function store(Request $request)
    {
        if ($response = $this->paytypeService->createPaytype($request->all())) {
            return $this->apiResponse(200, 'success', null, new PaytypeResource($response));
        }
        $errors = $this->paytypeService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function update(Request $request)
    {

        if ($response = $this->paytypeService->updatePaytype($request->all())) {
            return $this->apiResponse(200, 'success', null, new PaytypeResource($response));
        }
        $errors = $this->paytypeService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->paytypeService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Paytype deleted successfully', null, null);
        }
        $errors = $this->paytypeService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function SaveTranslation(Request $request, $id)
    {
        if (! $this->paytypeTranslationService->save($request->all(), $id)) {
            $errors = $this->paytypeTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->paytypeService->getByidWithTranslation($id)) {
            $collection = new PaytypeLangResources($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }

        return $this->apiResponse(400, 'fail', $this->paytypeService->errors(), null);
    }

    public function delLang($id)
    {
        if ($this->paytypeTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->paytypeTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $paytype = $this->paytypeService->getPaytypeById($id);
        if (! $paytype) {
            $errors = $this->paytypeService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->paytypeService->getAllChanges($paytype);

        return $this->apiResponse(200, 'success', null, PaytypeHistoryResource::collection($changes));
    }

    public function getToSupplierPaid()
    {
        $paytypes = $this->paytypeService->getTOSupplier();

        return $this->apiResponse(200, 'success', null, new PaytypeCollection($paytypes));
    }
}
