<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\AccountsStructure;
use App\Models\Accounting\TreeAccounting;

class TreeAccountingRepository
{
    public static function getTreeByCurrency(array $currencyIds = [])
    {
        return TreeAccounting::with('currency')
            ->whereHas('currency', function ($query) use ($currencyIds) {
                $query->where('status', 1);
                if (! empty($currencyIds)) {
                    $query->whereIn('id', $currencyIds);
                }
            })->where('active', 1)->get()->groupBy(fn ($account) => $account->currency?->id);
    }

    public function getByCurrency($id)
    {
        return TreeAccounting::with(['currency.currentTranslation', 'currentTranslation'])
            ->where('active', 1)
            ->where('currency_id', $id)
            ->select('id', 'title', 'currency_id')
            ->get();
    }

    public function getBanckByCurrency($id, $ids)
    {
        return TreeAccounting::with(['currency.currentTranslation', 'currentTranslation'])
            ->where('active', 1)
            ->whereIn('id', $ids)
            ->where('currency_id', $id)
            ->select('id', 'title', 'currency_id')
            ->get();
    }

    public function getAll()
    {
        return TreeAccounting::with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure', 'getAllChildren'])
            ->withCount('children')->where('tree_accounting_id', 0)->orderBy('serial_number', 'asc')->get();
    }

    public function create($data)
    {
        $tree = new TreeAccounting;
        $tree->title = $data['title'] ?? null;
        $tree->description = $data['description'] ?? null;
        $tree->active = $data['active'] ?? 1;
        $tree->tree_accounting_id = $data['parent_id'] ?? 0;
        $tree->accounts_structure_id = $data['accounts_structure_id'] ?? 1;
        $tree->serial_number = intval(str_replace('-', '', $data['serial_number_dight'])) ?? null;
        $tree->serial_number_dight = $data['serial_number_dight'] ?? null;
        $getlevel = $this->level($data['accounts_structure_id']);
        $tree->along_level = $getlevel->field_length ?? 1;
        $tree->the_level = $data['accounts_structure_id'];
        $tree->currency_id = $data['currency_id'] ?? null;
        $tree->company_id = $data['company_id'] ?? null;
        $tree->credit = $data['credit'] ?? 0;
        $tree->debit = $data['debit'] ?? 0;
        $tree->save();

        return $tree;
    }

    public function updated($data)
    {
        $tree = TreeAccounting::find($data['id']);
        $tree->title = $data['title'] ?? $tree->title;
        $tree->description = $data['description'] ?? $tree->title;
        $tree->active = $data['active'] ?? $tree->active;
        $tree->save();

        return $tree;
    }

    public function level($id)
    {
        return AccountsStructure::find($id);
    }

    public function findById($id)
    {
        return TreeAccounting::with(['translations', 'currentTranslation'])->find($id);
    }

    public static function CheckTree($ids)
    {
        return TreeAccounting::whereIn('id', $ids)->pluck('id')->toArray();
    }

    public function updateTitleAndDescription(int $id, string $title, ?string $description)
    {
        $tree = TreeAccounting::find($id);
        if (! $tree) {
            return null;
        }
        $tree->title = $title;
        $tree->description = $description ?? '';
        $tree->save();

        return $tree;
    }

    public function getParents()
    {
        return TreeAccounting::query()
            ->with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure'])
            ->withCount('children')
           // ->where('active', 1)
            ->where('tree_accounting_id', 0)
            ->orderBy('serial_number', 'asc')
            ->get();
    }

    public function getActive()
    {
        return TreeAccounting::with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure'])
            ->withCount('children')
            ->whereNotNull('currency_id')
            ->where('active', 1)
            ->orderBy('serial_number', 'asc')
            ->get();
    }

    public function getByParentId(int $parentId)
    {
        return TreeAccounting::query()
            ->with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure'])
            ->withCount('children')
           // ->where('active', 1)
            ->where('tree_accounting_id', $parentId)
            ->orderBy('serial_number', 'asc')
            ->get();
    }

    public function getList()
    {
        return TreeAccounting::query()
            ->with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure'])
            ->withCount('children')
            ->where('active', 1)
            ->orderBy('serial_number', 'asc')
            ->get();
    }

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $title = request('title', null);
        $limit = request('limit', 15);

        $query = TreeAccounting::query()
            ->when($title, function ($q, $title) {
                $q->where(function ($qq) use ($title) {
                    $qq->where('title', 'LIKE', "%$title%")
                        ->orWhereHas('translations', function ($tq) use ($title) {
                            $tq->where('title', 'LIKE', "%$title%")
                                ->where('lang_id', app('lang_id'));
                        });
                });
            })
            ->with(['currentTranslation', 'currency.currentTranslation', 'company.currentTranslation', 'parent.currentTranslation', 'structure'])
            ->withCount('children');

        if ($column === 'title') {
            $query->leftJoin('tree_accounting_translation', function ($join) {
                $join->on('tree_accounting.id', '=', 'tree_accounting_translation.tree_accounting_id')
                    ->where('tree_accounting_translation.lang_id', app('lang_id'));
            })->orderByRaw('COALESCE(tree_accounting_translation.title, tree_accounting.title) '.$order)
                ->select('tree_accounting.*');
        } elseif (in_array($column, ['id', 'active', 'created', 'modified'])) {
            $query->orderBy($column, $order);
        }

        return $query->orderBy('serial_number', 'asc')->paginate($limit);
    }

    /**
     * Get maximum second part for a given first part
     */
    public function getMaxSecondPart(string $firstPart): int
    {
        $maxSerial = TreeAccounting::where('serial_number_dight', 'LIKE', $firstPart.'-%')
            ->where('accounts_structure_id', 2)
            ->where('active', 1)
            ->max('serial_number_dight');

        if (! $maxSerial) {
            return 0;
        }

        $parts = explode('-', $maxSerial);

        return intval($parts[1]);
    }

    /**
     * Get maximum third part for given first and second parts
     */
    public function getMaxThirdPart(string $firstPart, string $secondPart): int
    {
        $pattern = $firstPart.'-'.$secondPart.'-%';
        $maxSerial = TreeAccounting::where('serial_number_dight', 'LIKE', $pattern)
            ->where('accounts_structure_id', 3)
            ->where('active', 1)
            ->max('serial_number_dight');

        if (! $maxSerial) {
            return 0;
        }

        $parts = explode('-', $maxSerial);

        return intval($parts[2]);
    }

    /**
     * Get maximum fourth part for given first, second, and third parts
     */
    public function getMaxFourthPart(string $firstPart, string $secondPart, string $thirdPart): int
    {
        $pattern = $firstPart.'-'.$secondPart.'-'.$thirdPart.'-%';
        $maxSerial = TreeAccounting::where('serial_number_dight', 'LIKE', $pattern)
            ->where('accounts_structure_id', 4)
            ->where('active', 1)
            ->max('serial_number_dight');

        if (! $maxSerial) {
            return 0;
        }

        $parts = explode('-', $maxSerial);

        return intval($parts[3]);
    }

    /**
     * Get maximum fifth part for given first, second, third, and fourth parts
     */
    public function getMaxFifthPart(string $firstPart, string $secondPart, string $thirdPart, string $fourthPart): int
    {
        $pattern = $firstPart.'-'.$secondPart.'-'.$thirdPart.'-'.$fourthPart.'-%';
        $maxSerial = TreeAccounting::where('serial_number_dight', 'LIKE', $pattern)
            ->where('accounts_structure_id', 5)
            ->where('active', 1)
            ->max('serial_number_dight');

        if (! $maxSerial) {
            return 0;
        }

        $parts = explode('-', $maxSerial);

        return intval($parts[4]);
    }
}
