<?php

namespace App\sys\Repository\Profile;

use App\Models\Profile\AccommodationReservation;
use App\Models\Profile\Attachment;

class AccommodationReservationRepository
{
    private $columns = [
        'id' => 'id',
        'profile_id' => 'profile_id',
        'accommodation_id' => 'accommodation_id',
        'supplier_id' => 'supplier_id',
        'city_id' => 'city_id',
        'country_id' => 'country_id',
        'type' => 'type',
        'status' => 'status',
        'confirmation_date' => 'confirmation_date',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated(array $filters = [])
    {
        $query = AccommodationReservation::with(['profile', 'accommodation']);

        if (! empty($filters['profile_id'])) {
            $query->where('profile_id', $filters['profile_id']);
        }
        if (! empty($filters['accommodation_id'])) {
            $query->where('accommodation_id', $filters['accommodation_id']);
        }
        if (! empty($filters['supplier_id'])) {
            $query->where('supplier_id', $filters['supplier_id']);
        }
        if (! empty($filters['city_id'])) {
            $query->where('city_id', $filters['city_id']);
        }
        if (! empty($filters['country_id'])) {
            $query->where('country_id', $filters['country_id']);
        }
        if (! empty($filters['type'])) {
            $query->where('type', $filters['type']);
        }
        if (! empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }
        if (! empty($filters['confirmation_date_from'])) {
            $query->whereDate('confirmation_date', '>=', $filters['confirmation_date_from']);
        }
        if (! empty($filters['confirmation_date_to'])) {
            $query->whereDate('confirmation_date', '<=', $filters['confirmation_date_to']);
        }

        $column = $filters['sort_by'] ?? null;
        $order = $filters['sort_order'] ?? 'asc';
        $limit = $filters['limit'] ?? 15;

        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        }

        return $query->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return AccommodationReservation::with(['profile', 'accommodation.currentTranslation'])->find($id);
    }

    public function create(array $data)
    {
        $item = new AccommodationReservation;
        $item->profile_id = $data['profile_id'];
        $item->accommodation_id = $data['accommodation_id'];
        $item->type = $data['type'];
        $item->status = 'pending';
        $item->confirmation_date = $data['confirmation_date'] ?? null;
        $item->confirmation_num = $data['confirmation_num'] ?? null;
        $item->supplier_id = $data['supplier_id'] ?? null;
        $item->city_id = $data['city_id'] ?? null;
        $item->country_id = $data['country_id'] ?? null;
        $item->customer_note = $data['customer_note'] ?? null;
        $item->hotel_note = $data['hotel_note'] ?? null;
        $item->reservation_num = $data['reservation_num'] ?? null;
        $item->reservation_link = $data['reservation_link'] ?? null;
        $item->count_rooms = $data['count_rooms'] ?? 0;
        $item->save();

        return $item->load(['profile', 'accommodation']);
    }

    public function update(AccommodationReservation $item, array $data)
    {
        $item->profile_id = $data['profile_id'] ?? $item->profile_id;
        $item->accommodation_id = $data['accommodation_id'] ?? $item->accommodation_id;
        $item->type = $data['type'] ?? $item->type;
        $item->status = $data['status'] ?? $item->status;
        $item->confirmation_date = $data['confirmation_date'] ?? $item->confirmation_date;
        $item->confirmation_num = $data['confirmation_num'] ?? $item->confirmation_num;
        $item->supplier_id = $data['supplier_id'] ?? $item->supplier_id;
        $item->city_id = $data['city_id'] ?? $item->city_id;
        $item->country_id = $data['country_id'] ?? $item->country_id;
        $item->customer_note = $data['customer_note'] ?? $item->customer_note;
        $item->hotel_note = $data['hotel_note'] ?? $item->hotel_note;
        $item->reservation_num = $data['reservation_num'] ?? $item->reservation_num;
        $item->reservation_link = $data['reservation_link'] ?? $item->reservation_link;
        $item->count_rooms = $data['count_rooms'] ?? $item->count_rooms;
        $item->save();

        return $item->load(['profile', 'accommodation.currentTranslation']);
    }

    public function del(array $ids)
    {
        return AccommodationReservation::whereIn('id', $ids)->delete();
    }

    public function getByAccommodationType(string $type)
    {
        return AccommodationReservation::whereHas('accommodation', function ($q) use ($type) {
            $q->where('type', $type);
        })->orderByDesc('updated_at')->with(['profile', 'accommodation.currentTranslation'])->get();
    }

    public function getByAccommodationTypeAndProfile(string $type, int $profileId)
    {
        return AccommodationReservation::where('profile_id', $profileId)
            ->whereHas('accommodation', function ($q) use ($type) {
                $q->where('type', $type);
            })
            ->with(['profile', 'accommodation'])
            ->withMin('rooms', 'check_in')
            ->withMax('rooms', 'check_out')
            ->orderByDesc('updated_at')
            ->get();
    }

    public function uploadAttachments(array $data)
    {
        $attachment = new Attachment;
        $attachment->profile_id = $data['profile_id'];
        $attachment->attachable_type = $data['attachable_type'];
        $attachment->attachable_id = $data['attachable_id'];
        $attachment->name = $data['name'] ?? null;
        $attachment->path = $data['path'];
        $attachment->save();

        return $attachment;
    }

    public function deleteAttachments(array $ids)
    {
        return Attachment::whereIn('id', $ids)->delete();
    }

    public function getAttachmentsByProfileId(int $profileId)
    {
        return Attachment::where('profile_id', $profileId)->get();
    }

    public function getAttachmentsByReservationId(int $reservationId)
    {
        return Attachment::where('attachable_type', (new AccommodationReservation)->getMorphClass())
            ->where('attachable_id', $reservationId)
            ->get();
    }

    public function findAttachmentsByIds(array $ids)
    {
        return Attachment::whereIn('id', $ids)->get();
    }

    public function checkReservation($type, $profile)
    {
        return AccommodationReservation::whereHas('accommodation', function ($q) use ($type) {
            $q->where('type', $type);
        })->where('profile_id', $profile)->exists();
    }

    public function getReservationsByProfile($type, $profile)
    {
        return AccommodationReservation::with('accommodation.currentTranslation', 'city', 'rooms', 'rooms.mealPlan', 'rooms.roomType', 'rooms.roomView')->whereHas('accommodation', function ($q) use ($type) {
            $q->where('type', $type);
        })->where('profile_id', $profile)->get();
    }
}
