<?php

namespace App\sys\Services\Accounting;

use App\Models\Accounting\Invoice;
use App\Models\Accounting\Paytype;
use App\Models\General\CompanyAccountMappings;
use App\Models\General\Currency;
use App\Models\General\TaxRateMappings;
use App\Models\Profile\Profile;
use App\Models\Profile\ProfileTraveler;
use App\Models\Profile\ProfileTravelerTax;
use App\Models\UserAccountMapping;
use App\sys\Enums\CompanyAccountMapping;
use App\sys\Enums\PaymentTypes;
use App\sys\Helper;
use App\sys\Repository\Accounting\ConstraintRepository;
use App\sys\Repository\Accounting\InvoiceRepository;
use App\sys\Repository\Accounting\PaytypeRepository;
use App\sys\Repository\Accounting\UnpostedCollectionsRepository;
use App\sys\Repository\General\CurrencyRepository;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Repository\Profile\TravelerRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UnpostedCollectionsServices extends Services
{
    private $unpostedRepo;

    private $profileAccount;

    private $payType;

    private $currency;

    private $invoces;

    private $invoiceRepository;

    public function __construct()
    {
        $this->unpostedRepo = new UnpostedCollectionsRepository;
        $this->profileAccount = new TravelerRepository;
        $this->payType = new PayTypeRepository;
        $this->currency = new CurrencyRepository;
        $this->invoces = new InvoiceServicesRepository;
        $this->invoiceRepository = new InvoiceRepository;
    }

    public function createInvoiceWithoutPayment(int $profileId, array $data = [])
    {
        $validator = Validator::make(
            array_merge($data, ['profile_id' => $profileId]),
            [
                'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            ]
        );

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $profile = Profile::find($profileId);
        if (! $profile) {
            $this->setError(['profile_id' => ['الملف غير موجود']]);

            return false;
        }

        if ($profile->has_invoice) {
            $this->setError(['invoice' => ['تم إنشاء فاتورة لهذا الملف بالفعل']]);

            return false;
        }

        $defaultCurrency = Currency::where('is_default', 1)->first();
        if (! $defaultCurrency) {
            $this->setError(['currency_id' => ['الرجاء تعيين العملة الافتراضية للنظام.']]);

            return false;
        }

        $totalInDefaultCurrency = $this->calculateTotalFromTravelers($profile->id, $defaultCurrency);
        $data['amount'] = 0;
        $data['currency_id'] = $data['currency_id'] ?? $defaultCurrency->id;

        return $this->createInvoice(
            $profile,
            $data,
            $totalInDefaultCurrency,
            0,
            $totalInDefaultCurrency
        );
    }

    public function add($data)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],
            'pay_type_id' => ['required', 'integer', 'exists:pay_type,id'],
            'accounting_id' => ['nullable', 'integer', 'exists:pay_type_accounting,id'],
            'date' => ['required', 'date'],
            'next_date' => ['nullable', 'date', 'after:today'],
            'amount' => ['required', 'numeric', 'min:0'],
            'reference' => ['nullable'],
            'details' => ['nullable'],
            'customer_name' => ['nullable'],
        ];
        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
            $amount = $this->unpostedRepo->getAmountForCurrnecy($data['profile_id'], $data['currency_id']);
            $finalAmount = $amountprofile - $amount;
            if ($finalAmount < $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than currency amount');
            }
        });
        // تحديد إعدادات طريقة الدفع:
        // - لو مبعوت accounting_id نستخدمه مباشرة
        // - غير كده نستخدم الإعداد الافتراضي حسب العملة
        $paytype = $this->payType->getPayType($data['pay_type_id']);

        $pay = in_array($paytype->type, ['bank', 'payment_method_rates']) ? $this->payType->getaccount($data['accounting_id']) : [];

        if ($paytype->type == 'payment_method_rates' && ! empty($pay)) {
            if ($pay->max_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than max limit');
            }
            if ($pay->min_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than min limit');
            }
        }
        if (in_array($paytype->type, ['bank', 'payment_method_rates']) && empty($pay)) {
            $validator->errors()->add('accounting_id', 'It cannot be empty');
        }

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        if ($paytype->type == 'payment_method_rates') {
            $data['fee_amount'] = $data['amount'] * ($pay->percentage / 100);
            $data['transaction_amount'] = $data['amount'] - $data['fee_amount'];
        } else {
            $data['transaction_amount'] = $data['amount'];
            $data['fee_amount'] = 0;
        }
        // get Currnecy rate
        $data['currency_rate'] = $this->currency->findByIdOrFail($data['currency_id'])?->exchange_rate ?? null;

        $unpost = $this->unpostedRepo->add($data);

        return $this->constraints($unpost, $pay, $data, $paytype) !== false;
    }

    private function constraints($unpost, $pay, $data, ?Paytype $paytypeModel = null)
    {
        $unpost->load('profile');
        $profile = $unpost->profile;

        $defaultCurrency = Currency::where('is_default', 1)->first();
        if (! $defaultCurrency) {
            return;
        }

        $totalInDefaultCurrency = $this->calculateTotalFromTravelers($profile->id, $defaultCurrency);
        $paidAmountInDefaultCurrency = $this->convertToDefaultCurrency(
            (float) ($data['amount'] ?? 0),
            $data['currency_id'],
            $defaultCurrency
        );
        $remainingAmount = $totalInDefaultCurrency - $paidAmountInDefaultCurrency;

        if ($profile->has_invoice == 0) {
            $invoice = $this->createInvoice($profile, $data, $totalInDefaultCurrency, $paidAmountInDefaultCurrency, $remainingAmount);
            if (! $invoice) {
                return false;
            }
        } else {
            $invoice = $this->updateInvoice($profile, $data, $paidAmountInDefaultCurrency);
            if (! $invoice) {
                return false;
            }
        }
        //
        if ($data['amount'] == 0) {
            return true;
        }
        // اضافه سند قبض
        // حساب آجل العملاء (القيد المقابل للعميل)
        $customerTree = Helper::getCompanyTreeAccount($profile->company_id, CompanyAccountMapping::CUSTOMER_CREDIT->value, [$data['currency_id']]);

        $paytypeType = $paytypeModel?->type;
        $transfers = [];

        if (! empty($pay)) {
            // لو نوع الدفع بنك: مفيش عمولة، نستخدم حساب التحويل بس
            if ($paytypeType === PaymentTypes::BANK->value) {
                $transfers[] = [
                    'tree_accounting_id' => $pay->tree_accounting_transfer_id,
                    'currency_id' => $data['currency_id'],
                    'invoice_id' => $invoice->id,
                    'name' => 'transfer',
                    'debit' => $data['transaction_amount'] * $data['currency_rate'],
                    'creditor' => 0,
                    'currency_debit' => $data['transaction_amount'],
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                ];
            } else {
                // نسب طرق الدفع (أو أي نوع له عمولة): نفس المنطق القديم
                $transfers = [[
                    'tree_accounting_id' => $pay->tree_accounting_transfer_id,
                    'currency_id' => $data['currency_id'],
                    'invoice_id' => $invoice->id,
                    'name' => 'transfer',
                    'debit' => $data['transaction_amount'] * $data['currency_rate'],
                    'creditor' => 0,
                    'currency_debit' => $data['transaction_amount'],
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                ], [

                    'tree_accounting_id' => $pay->tree_account_commission_id,
                    'currency_id' => $data['currency_id'],
                    'invoice_id' => $invoice->id,
                    'name' => 'commission',
                    'debit' => $data['fee_amount'] * $data['currency_rate'],
                    'creditor' => 0,
                    'currency_debit' => $data['fee_amount'],
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                ]];
            }
        } else {
            // لو مفيش إعداد محاسبي:
            // - company_cash: نستخدم حساب خزنة الشركة (نفس القديم)
            // - employee_cash: نستخدم حساب المحفظة من user_account_mappings
            if ($paytypeType === PaymentTypes::EMPLOYEE_CASH->value) {
                $walletMapping = UserAccountMapping::where('user_id', Auth::id())
                    ->where('currency_id', $data['currency_id'])
                    ->first();

                if (! $walletMapping || ! $walletMapping->wallet_account_id) {
                    $this->setError(['accounting' => ['برجاء تعيين حساب المحفظة للموظف لهذه العملة.']]);

                    return false;
                }

                $transfers[] = [
                    'tree_accounting_id' => $walletMapping->wallet_account_id,
                    'currency_id' => $data['currency_id'],
                    'name' => 'employee_wallet',
                    'invoice_id' => $invoice->id,
                    'debit' => $data['amount'] * $data['currency_rate'],
                    'creditor' => 0,
                    'currency_debit' => $data['amount'],
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                ];
            } else {
                $cashTree = Helper::getCompanyTreeAccount($profile->company_id, CompanyAccountMapping::CACH->value, [$data['currency_id']]);

                $transfers[] = [
                    'tree_accounting_id' => $cashTree[$data['currency_id']] ?? null,
                    'currency_id' => $data['currency_id'],
                    'name' => 'CACH',
                    'invoice_id' => $invoice->id,
                    'debit' => $data['amount'] * $data['currency_rate'],
                    'creditor' => 0,
                    'currency_debit' => $data['amount'],
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                ];
            }
        }
        $transfers[] = [
            'tree_accounting_id' => $customerTree[$data['currency_id']] ?? null,
            'currency_id' => $data['currency_id'],
            'name' => 'customer_credit_',
            'invoice_id' => $invoice->id,
            'debit' => 0,
            'creditor' => $data['amount'] * $data['currency_rate'],
            'currency_debit' => 0,
            'currency_creditor' => $data['amount'],
            'cost_center_id' => null,
        ];
        $total_debit = array_sum(array_column($transfers, 'debit'));
        $total_creditor = array_sum(array_column($transfers, 'creditor'));
        $const = [
            'customer_id' => $profile->customer_id,
            'profile_id' => $profile->id,
            'invoice_id' => $invoice->id,
            'sales_id' => $profile->user_id,
            'date' => $data['date'] ?? now()->format('Y-m-d'),
            'total_creditor' => $total_creditor,
            'total_debit' => $total_debit,
            'type_optional' => 'invoice',
            'name' => 'دفع عميل علي ملف',
            'description' => 'دفع عميل _'.$profile->profile_number,
            'active' => 1,
            'capture_exchange' => 'receipt',
            'creation_mode' => 'automatic',
            'currency_id' => $data['currency_id'],
            'currency_transfer_rate' => $data['currency_rate'],
            'company_id' => $profile->company_id,
            'transfers' => $transfers,

        ];
        $constRepo = new ConstraintRepository;
        $constRepo->create($const, 'yes');

        return true;
    }

    public function update($data)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],
            'pay_type_id' => ['required', 'integer', 'exists:pay_type,id'],
            'date' => ['required', 'date'],
            'next_date' => ['nullable', 'date', 'after:today'],
            'amount' => ['required', 'numeric', 'min:0'],
            'reference' => ['nullable'],
            'details' => ['nullable'],
            'customer_name' => ['nullable'],
        ];
        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
            $amount = $this->unpostedRepo->getAmountForCurrnecyIgonrId($data['id'], $data['profile_id'], $data['currency_id']);
            $finalAmount = $amountprofile - $amount;
            if ($finalAmount < $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than currency amount');
            }
        });
        $pay = $this->payType->getpayTypeAccountingByCurrency($data['pay_type_id'], $data['currency_id']);
        if (! empty($pay)) {
            if ($pay->max_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than max limit');
            }

            if ($pay->min_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than min limit');
            }
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        if (! empty($pay)) {
            $data['fee_amount'] = $data['amount'] * ($pay->percentage / 100);
            $data['transaction_amount'] = $data['amount'] - $data['fee_amount'];
        } else {
            $data['transaction_amount'] = $data['amount'];
            $data['fee_amount'] = 0;
        }
        $data['currency_rate'] = $this->currency->findByIdOrFail($data['currency_id'])?->exchange_rate ?? null;

        return $this->unpostedRepo->updated($data);
    }

    public function getByProfile($profile_id)
    {
        return $this->unpostedRepo->getByProfileId($profile_id);
    }

    public function getAmount($data)
    {
        $rules = [
            'id' => ['nullable', 'integer', 'exists:unposted_collections,id'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],

        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
        $amount = $data['id'] != null ? $this->unpostedRepo->getAmountForCurrnecyIgonrId($data['id'], $data['profile_id'], $data['currency_id']) : $this->unpostedRepo->getAmountForCurrnecy($data['profile_id'], $data['currency_id']);

        return [

            'remaining_amount' => (float) number_format($amountprofile - $amount, 3, '.', ''),
            'profile_amount' => (float) number_format($amountprofile, 3, '.', ''),
            'current_amount' => (float) number_format($amount, 3, '.', ''),
        ];
    }

    public function summary($profile_id)
    {
        // التكلفه
        $cost = $this->profileAccount->getByprofile($profile_id);
        // currency
        $currency = $cost->pluck('currency_id')->toArray();
        // المحصل
        $collector = $this->unpostedRepo->getPostedByCurrnecy($profile_id);
        // المدفوع final_currency_id
        $inovesIn = $this->invoces->summeryinCurrency($profile_id, $currency);
        $data = [];
        foreach ($currency as $key) {
            $currency_name = $this->currency->findByIdOrFail($key);
            $costBycurrency = $cost->where('currency_id', $key)->first();
            $collectorBycurrency = $collector->where('currency_id', $key)->first();
            $purchase = $inovesIn->where('final_currency_id', $key)->first();
            $data[] = [
                'currency_id' => $key,
                'currency_name' => $currency_name->name ?? null,
                'cost' => $costBycurrency->total ?? 0,
                'collector' => $collectorBycurrency->total_amount ?? 0,
                'purchase_price' => $purchase->grand_total ?? 0,
                'remaining' => ($costBycurrency->total ?? 0) - ($collectorBycurrency->total_amount ?? 0),
                'profit' => ($costBycurrency->total ?? 0) - ($purchase->grand_total ?? 0),
                'currency_type' => 'mine_profile',
            ];
        }
        $inovesOut = $this->invoces->summeryinCurrencyNotIn($profile_id, $currency);
        if (! empty($inovesOut)) {
            foreach ($inovesOut as $key) {
                $data[] = [
                    'currency_id' => (int) $key->final_currency_id,
                    'currency_name' => $key->currency_name,
                    'cost' => 0,
                    'collector' => 0,
                    'purchase_price' => $key->grand_total ?? 0,
                    'remaining' => 0,
                    'profit' => 0,
                    'currency_type' => 'out_profile',
                ];
            }
        }

        return $data;
    }

    public function selectedCurrency($profile_id)
    {
        return $this->profileAccount->selectedCurencyToProfile($profile_id);
    }

    /**
     * Calculate total amount from travelers in default currency
     */
    private function calculateTotalFromTravelers($profileId, $defaultCurrency)
    {
        $travelers = ProfileTraveler::where('profile_id', $profileId)->get();
        $total = 0.0;

        foreach ($travelers as $traveler) {
            $travelerTotal = (float) ($traveler->total_price ?? 0);
            $travelerCurrency = Currency::find($traveler->currency_id);
            if ($travelerCurrency) {
                if ($traveler->currency_id == $defaultCurrency->id) {
                    $total += $travelerTotal;
                } else {
                    $exchangeRate = $travelerCurrency->exchange_rate / $defaultCurrency->exchange_rate;
                    $total += $travelerTotal * $exchangeRate;
                }
            }
        }

        return $total;
    }

    /**
     * Convert amount to default currency
     */
    private function convertToDefaultCurrency($amount, $currencyId, $defaultCurrency)
    {
        $currency = Currency::find($currencyId);
        if (! $currency) {
            return 0.0;
        }

        if ($currencyId == $defaultCurrency->id) {
            return $amount;
        }

        $exchangeRate = $currency->exchange_rate / $defaultCurrency->exchange_rate;

        return $amount * $exchangeRate;
    }

    /**
     * Create new invoice for profile
     */
    private function createInvoice($profile, $data, $totalInDefaultCurrency, $paidAmountInDefaultCurrency, $remainingAmount)
    {
        $invoiceData = [
            'profile_id' => $profile->id,
            'customer_id' => $profile->customer_id,
            'company_id' => $profile->company_id,
            'invoice_date' => $data['date'] ?? now()->format('Y-m-d'),
            'recip_date' => $data['date'] ?? now()->format('Y-m-d'),
            'invoice_date_due' => null,
            'invoice_price' => round($totalInDefaultCurrency, 2),
            'paid_price' => round($paidAmountInDefaultCurrency, 2),
            'remaining_price' => round($remainingAmount, 2),
            'invoice_type' => 1,
            'paid_date' => $data['date'] ?? now()->format('Y-m-d'),
            'next_paid_date' => $data['next_date'] ?? null,
        ];

        $invoice = $this->invoiceRepository->create($invoiceData);

        $profile->has_invoice = 1;
        $profile->save();

        \App\Models\invoice\InvoiceServices::where('profile_id', $profile->id)
            ->whereNull('invoice_id')
            ->update(['invoice_id' => $invoice->id, 'travel_tourism_steps' => 1]);

        if ($this->createCustomerCreditConstraints($profile, $invoice) === false) {
            return false;
        }

        return $invoice;
    }

    private function createCustomerCreditConstraints($profile, $invoice): bool
    {
        $defaultCurrency = Currency::where('is_default', 1)->first();
        if (! $defaultCurrency) {
            $this->setError(['currency_id' => ['الرجاء تعيين العملة الافتراضية للنظام.']]);

            return false;
        }

        $travelers = ProfileTraveler::selectRaw('currency_id, SUM(total_price) as total_price_sum, SUM(net_total) as net_total_sum')
            ->where('profile_id', $profile->id)
            ->groupBy('currency_id')
            ->get();

        if ($travelers->isEmpty()) {
            return true;
        }

        $taxAggregates = ProfileTravelerTax::query()
            ->join('pr_profile_travelers as pt', 'pt.id', '=', 'pr_profile_travelers_tax.profile_travelers_id')
            ->selectRaw('pt.currency_id, pr_profile_travelers_tax.tax_rate_id, SUM(pr_profile_travelers_tax.amount) as tax_sum')
            ->where('pr_profile_travelers_tax.profile_id', $profile->id)
            ->groupBy('pt.currency_id', 'pr_profile_travelers_tax.tax_rate_id')
            ->get()
            ->groupBy('currency_id');

        $constraintRepo = new ConstraintRepository;
        $transfers = [];

        foreach ($travelers as $row) {
            $currencyId = (int) $row->currency_id;
            $totalPrice = (float) ($row->total_price_sum ?? 0);
            $netTotal = (float) ($row->net_total_sum ?? 0);

            if ($totalPrice <= 0) {
                continue;
            }

            $currency = Currency::find($currencyId);
            if (! $currency) {
                $this->setError(['currency_id' => ['لم يتم العثور على العملة المحددة للقيد.']]);

                return false;
            }

            $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);

            $customerAccount = CompanyAccountMappings::where('company_id', $profile->company_id)
                ->where('currency_id', $currencyId)
                ->where('type', CompanyAccountMapping::CUSTOMER_CREDIT->value)
                ->first();

            if (! $customerAccount) {
                $this->setError(['accounting' => ['برجاء تعيين حساب آجل العملاء للعملة '.$currency->code]]);

                return false;
            }

            $revenueAccount = CompanyAccountMappings::where('company_id', $profile->company_id)
                ->where('currency_id', $currencyId)
                ->where('type', CompanyAccountMapping::PACKAGE_SERVICE_REVENUE->value)
                ->first();

            if (! $revenueAccount) {
                $this->setError(['accounting' => ['برجاء تعيين حساب إيرادات الباقات للعملة '.$currency->code]]);

                return false;
            }

            $taxRows = $taxAggregates->get($currencyId, collect());
            $taxTotal = (float) $taxRows->sum('tax_sum');

            if ($netTotal <= 0) {
                $netTotal = max(0, $totalPrice - $taxTotal);
            }

            $customerDebitDefault = round($totalPrice * $exchangeRate, 2);
            $currencyTransfers = [[
                'tree_accounting_id' => $customerAccount->tree_account_id,
                'currency_id' => $currencyId,
                'name' => 'قيد آجل عملاء - ملف '.$profile->profile_number,
                'debit' => $customerDebitDefault,
                'creditor' => 0,
                'currency_debit' => round($totalPrice, 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
            ]];

            $revenueCreditDefault = round($netTotal * $exchangeRate, 2);
            $revenueTransfer = [
                'tree_accounting_id' => $revenueAccount->tree_account_id,
                'currency_id' => $currencyId,
                'name' => 'إيراد خدمات - ملف '.$profile->profile_number,
                'debit' => 0,
                'creditor' => $revenueCreditDefault,
                'currency_debit' => 0,
                'currency_creditor' => round($netTotal, 2),
                'cost_center_id' => null,
            ];

            $totalCreditDefault = $revenueCreditDefault;

            foreach ($taxRows as $taxRow) {
                $taxAmount = (float) ($taxRow->tax_sum ?? 0);
                if ($taxAmount <= 0) {
                    continue;
                }

                $taxMapping = TaxRateMappings::where('tax_rate_id', $taxRow->tax_rate_id)
                    ->where('currency_id', $currencyId)
                    ->where('type', 'sales')
                    ->first();

                if (! $taxMapping) {
                    $this->setError(['tax' => ['برجاء تعيين حساب ضريبة المبيعات للضريبة رقم '.$taxRow->tax_rate_id.' للعملة '.$currency->code]]);

                    return false;
                }

                $taxDefault = round($taxAmount * $exchangeRate, 2);
                $totalCreditDefault += $taxDefault;

                $currencyTransfers[] = [
                    'tree_accounting_id' => $taxMapping->tree_account_id,
                    'currency_id' => $currencyId,
                    'name' => 'ضريبة مبيعات - ملف '.$profile->profile_number,
                    'debit' => 0,
                    'creditor' => $taxDefault,
                    'currency_debit' => 0,
                    'currency_creditor' => round($taxAmount, 2),
                    'cost_center_id' => null,
                ];
            }

            $difference = $customerDebitDefault - $totalCreditDefault;
            if (abs($difference) > 0.02) {
                $this->setError(['constraint' => ['قيمة القيد غير متوازنة للعملة '.$currency->code]]);

                return false;
            }

            if (abs($difference) > 0) {
                $revenueTransfer['creditor'] = round($revenueTransfer['creditor'] + $difference, 2);
            }

            $currencyTransfers[] = $revenueTransfer;

            foreach ($currencyTransfers as $transfer) {
                $transfers[] = $transfer;
            }
        }

        if (empty($transfers)) {
            return true;
        }

        $totalDebitDefault = array_sum(array_map(static fn ($transfer) => $transfer['debit'], $transfers));
        $totalCreditDefault = array_sum(array_map(static fn ($transfer) => $transfer['creditor'], $transfers));

        $constraintData = [
            'customer_id' => $profile->customer_id,
            'profile_id' => $profile->id,
            'invoice_id' => $invoice->id,
            'sales_id' => $profile->user_id,
            'date' => $invoice->invoice_date ?? now()->format('Y-m-d'),
            'total_creditor' => $totalCreditDefault,
            'total_debit' => $totalDebitDefault,
            'name' => 'قيد آجل عملاء',
            'description' => 'قيد آجل عملاء للملف رقم '.$profile->profile_number,
            'active' => 1,
            'capture_exchange' => 'constraint',
            'creation_mode' => 'automatic',
            'type_optional' => 'start_invoice',
            'currency_id' => $defaultCurrency->id,
            'currency_transfer_rate' => 1,
            'company_id' => $profile->company_id,
            'transfers' => $transfers,
        ];

        $constraintRepo->create($constraintData, 'yes');

        return true;
    }

    /**
     * Update existing invoice
     */
    private function updateInvoice($profile, $data, $paidAmountInDefaultCurrency)
    {
        $invoice = Invoice::where('profile_id', $profile->id)->first();
        if (! $invoice) {
            return;
        }

        $currentPaid = (float) ($invoice->paid_price ?? 0);
        $currentRemaining = (float) ($invoice->remaining_price ?? 0);

        $updateData = [
            'paid_price' => round($currentPaid + $paidAmountInDefaultCurrency, 2),
            'remaining_price' => round($currentRemaining - $paidAmountInDefaultCurrency, 2),
            'paid_date' => $data['date'] ?? $invoice->paid_date,
        ];

        if (isset($data['next_date'])) {
            $updateData['invoice_date_due'] = $data['next_date'];
        }

        return $this->invoiceRepository->update($invoice, $updateData);
    }
}
