<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CityTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CityTranslationService extends Services
{
    private $cityTranslation;

    public function __construct()
    {
        $this->cityTranslation = new CityTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['city_id' => 'required|integer|exists:cities,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            // Validation rules
            $rules["$index.id"] = 'sometimes|nullable|exists:city_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('city_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('city_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';

            // Check for duplicated combination
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same City.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['city_id' => $id]), $rules);

        // Inject custom duplicate error if any
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->cityTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->cityTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:city_translation,id',
        ];
        $validator = \Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->cityTranslation->del($id);
    }
}
