<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CurrencyTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CurrencyTranslationService extends Services
{
    private $currencyTranslation;

    public function __construct()
    {
        $this->currencyTranslation = new CurrencyTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['currency_id' => 'required|integer|exists:currencies,id'];
        $combinations = [];
        $customErrors = [];
        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:currency_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('currency_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('currency_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:100';
            $rules["$index.symbol"] = 'nullable|string|max:5';
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Currency.';
                } else {
                    $combinations[] = $key;
                }
            }
        }
        $validator = Validator::make(array_merge($data, ['currency_id' => $id]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->currencyTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->currencyTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:currency_translation,id',
        ];
        $validator = \Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->currencyTranslation->del($id);
    }
}
