<?php

namespace App\sys\Services\Profile;

use App\Models\Accounting\Paytype;
use App\Models\UserAccountMapping;
use App\sys\Enums\PaymentTypes;
use App\sys\Repository\Profile\CustomerProfilesRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class CustomerProfilesService extends Services
{
    protected CustomerProfilesRepository $customerProfilesRepository;

    public function __construct(CustomerProfilesRepository $customerProfilesRepository)
    {
        $this->customerProfilesRepository = $customerProfilesRepository;
    }

    public function getCustomerProfiles($customerId)
    {
        // Define filter variables first
        $arrivalDateFrom = request('arrival_date_from');
        $arrivalDateTo = request('arrival_date_to');
        $departureDateFrom = request('departure_date_from');
        $departureDateTo = request('departure_date_to');

        // Validate input
        if (! $this->validateCustomerProfilesRequest($customerId, $arrivalDateFrom, $arrivalDateTo, $departureDateFrom, $departureDateTo)) {
            return false;
        }

        return $this->customerProfilesRepository->getCustomerSummary(
            $customerId,
            $arrivalDateFrom,
            $arrivalDateTo,
            $departureDateFrom,
            $departureDateTo
        );
    }

    public function getProfileDetails($profileId)
    {
        // Validate profile ID
        if (! $this->validateProfileId($profileId)) {
            return false;
        }

        $costs = $this->customerProfilesRepository->getProfileCostsByCurrency($profileId);
        $payments = $this->customerProfilesRepository->getProfilePaymentsByCurrency($profileId);

        return $this->mergeCurrencyData($costs, $payments);
    }

    /**
     * Validate customer profiles request
     */
    private function validateCustomerProfilesRequest($customerId, $arrivalDateFrom, $arrivalDateTo, $departureDateFrom, $departureDateTo)
    {
        $rules = [
            'customer_id' => 'required|integer|exists:pr_customer,id',
        ];

        $validator = Validator::make([
            'customer_id' => $customerId,
        ], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    /**
     * Validate profile ID
     */
    private function validateProfileId($profileId)
    {
        $rules = [
            'profile_id' => 'required|integer|exists:pr_profile,id',
        ];

        $validator = Validator::make(['profile_id' => $profileId], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    /**
     * Merge currency costs and payments data
     */
    private function mergeCurrencyData($costs, $payments)
    {
        $currencies = collect($costs)->merge($payments)->groupBy('currency_id');

        return $currencies->map(function ($currencyGroup) {
            $cost = $currencyGroup->where('total_cost', '>', 0)->first();
            $payment = $currencyGroup->where('total_paid', '>', 0)->first();

            return [
                'currency_id' => $currencyGroup->first()['currency_id'],
                'currency_name' => $currencyGroup->first()['currency_name'],
                'currency_code' => $currencyGroup->first()['currency_code'],
                'currency_symbol' => $currencyGroup->first()['currency_symbol'],
                'exchange_rate' => $currencyGroup->first()['exchange_rate'],
                'is_default' => $currencyGroup->first()['is_default'],
                'total_cost' => $cost['total_cost'] ?? 0,
                'total_paid' => $payment['total_paid'] ?? 0,
                'remaining' => ($cost['total_cost'] ?? 0) - ($payment['total_paid'] ?? 0),
            ];
        })->values()->toArray();
    }

    /**
     * Build customer account statement (by company/currency and date range)
     */
    public function getCustomerAccountStatement($data)
    {
        // Validate inputs (light)
        $rules = [
            'company_id' => 'nullable|exists:companies,id',
            'currency_id' => 'nullable|exists:currencies,id',
            'customer_id' => 'nullable|exists:pr_customer,id',
            'from' => 'nullable|date',
            'to' => 'nullable|date',
            'type_optional' => 'nullable|string', // تمت الإضافة هنا
        ];
        $validator = \Illuminate\Support\Facades\Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $companyId = $data['company_id'] ?? null;
        $currencyId = $data['currency_id'] ?? null;
        $customerId = $data['customer_id'] ?? null;
        $dateFrom = $data['from'] ?? null;
        $dateTo = $data['to'] ?? null;
        $typeOptional = $data['type_optional'] ?? null; // تم الإضافة

        // Delegate to repository for data fetching/assembly
        $data = $this->customerProfilesRepository->getCustomerAccountStatement(
            $companyId,
            $currencyId,
            $dateFrom,
            $dateTo,
            $customerId,
            $typeOptional
        );
        if ($data === null) {
            $this->setError(['message' => 'No customer credit account mapping found for the given company/currency']);

            return false;
        }

        return $data;
    }

    /**
     * Get customer financial summary by currency (aggregated across all profiles)
     */
    public function getCustomerFinancialSummary($customerId)
    {
        // Define filter variables first
        $arrivalDateFrom = request('arrival_date_from');
        $arrivalDateTo = request('arrival_date_to');
        $departureDateFrom = request('departure_date_from');
        $departureDateTo = request('departure_date_to');

        // Validate input
        if (! $this->validateCustomerProfilesRequest($customerId, $arrivalDateFrom, $arrivalDateTo, $departureDateFrom, $departureDateTo)) {
            return false;
        }

        return $this->customerProfilesRepository->getCustomerFinancialSummary(
            $customerId,
            $arrivalDateFrom,
            $arrivalDateTo,
            $departureDateFrom,
            $departureDateTo
        );
    }

    /**
     * Customer invoice balances grouped by customer (per company or all)
     */
    public function getCustomerInvoiceBalances($companyId)
    {
        $validator = Validator::make([
            'company_id' => $companyId,
        ], [
            'company_id' => 'nullable|integer|exists:companies,id',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $data = $this->customerProfilesRepository->getCustomerInvoiceBalances($companyId);
        if ($data === null) {
            $this->setError(['message' => 'No customer credit account mapping found for the given company/currency']);

            return false;
        }

        return $data;
    }

    public function wallet($data)
    {
        // 1. Validation كما طلب العميل
        $rules = [
            'customer_id' => ['required', 'integer', 'exists:pr_customer,id'],
            'company_id' => ['required', 'integer', 'exists:companies,id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'amount' => ['required', 'numeric'],
            'type' => ['required', 'in:receipt,payment'],
            'pay_type_id' => ['required', 'exists:pay_type,id'],
            'accounting_id' => ['nullable', 'integer', 'exists:pay_type_accounting,id'],
            'description' => ['nullable', 'string'],
            'date' => ['required', 'date'],
        ];
        $validator = Validator::make($data, $rules);

        $payType = PayType::find($data['pay_type_id']);
        if (in_array($payType->type, ['bank', 'payment_method_rates']) && $data['accounting_id'] == null) {
            $validator->errors()->add('accounting_id', 'It cannot be empty');
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        // 2. المتغيرات الأساسية
        $company_id = $data['company_id'];
        $currency_id = $data['currency_id'];
        $amount = (float) $data['amount'];
        $pay_type_id = $data['pay_type_id'];
        $type = $data['type'];
        $customer_id = $data['customer_id'];
        $date = $data['date'];
        $accounting_id = $data['accounting_id'] ?? null;
        $description = $data['description'] ?? null;

        // 3. جلب حساب العميل customer_advance
        $advance = \App\Models\General\CompanyAccountMappings::where([
            'company_id' => $company_id,
            'currency_id' => $currency_id,
            'type' => 'customer_advance',
        ])->first();
        if (! $advance) {
            $this->setError(['advance_account' => 'حساب advance غير موجود']);

            return false;
        }
        $advance_account = $advance->tree_account_id;
        $transfer_account = null;
        $commission_account = null;
        $commission_percentage = 0;
        $commission_amount = 0;
        $paytypeType = $payType->type;
        // 4. جلب الحساب المقابل & العمولة
        if ($paytypeType === PaymentTypes::COMPANY_CASH->value) {
            $cash = \App\Models\General\CompanyAccountMappings::where([
                'company_id' => $company_id,
                'currency_id' => $currency_id,
                'type' => 'cach',
            ])->first();
            if (! $cash) {
                $this->setError(['fallback_account' => 'لم يتم العثور على حساب cash كـ fallback']);

                return false;
            }
            $transfer_account = $cash->tree_account_id;
        } elseif ($paytypeType === PaymentTypes::BANK->value) {
            $payTypeRow = \App\Models\Accounting\PaytypeAccounting::find($accounting_id);
            $transfer_account = $payTypeRow->tree_accounting_transfer_id;
        } elseif ($paytypeType === PaymentTypes::PAYMENT_METHOD_RATES->value) {
            $payTypeRow = \App\Models\Accounting\PaytypeAccounting::find($accounting_id);
            $transfer_account = $payTypeRow->tree_accounting_transfer_id;
            $commission_account = $payTypeRow->tree_account_commission_id;
            $commission_percentage = (float) $payTypeRow->percentage;
            if ($commission_account && $commission_percentage > 0) {
                $commission_amount = round(($amount * $commission_percentage) / 100, 2);
            }
        } elseif ($paytypeType === PaymentTypes::EMPLOYEE_CASH->value) {
            $walletMapping = UserAccountMapping::where('user_id', Auth::id())
                ->where('currency_id', $data['currency_id'])
                ->first();

            if (! $walletMapping || ! $walletMapping->wallet_account_id) {
                $this->setError(['pay_type_id' => ['برجاء تعيين حساب المحفظة للموظف لهذه العملة.']]);

                return false;
            }
            $transfer_account = $walletMapping->wallet_account_id;
        }

        // 5. تجهيز ترتيب التحويلات (transfer array)
        $transfers = [];
        if ($type == 'receipt') {
            $transfers[] = [
                'tree_accounting_id' => $advance_account,
                'name' => 'wallet',
                'debit' => $amount,
                'creditor' => 0,
            ];
            $transfers[] = [
                'tree_accounting_id' => $transfer_account,
                'name' => 'wallet counterparty',
                'debit' => 0,
                'creditor' => ($amount - $commission_amount),
            ];
            if ($commission_amount > 0 && $commission_account) {
                $transfers[] = [
                    'tree_accounting_id' => $commission_account,
                    'name' => 'wallet commission',
                    'debit' => 0,
                    'creditor' => $commission_amount,
                ];
            }
        } else { // payment
            $transfers[] = [
                'tree_accounting_id' => $advance_account,
                'name' => 'wallet',
                'debit' => 0,
                'creditor' => $amount,
            ];
            $transfers[] = [
                'tree_accounting_id' => $transfer_account,
                'name' => 'wallet counterparty',
                'debit' => ($amount - $commission_amount),
                'creditor' => 0,
            ];
            if ($commission_amount > 0 && $commission_account) {
                $transfers[] = [
                    'tree_accounting_id' => $commission_account,
                    'name' => 'wallet commission',
                    'debit' => $commission_amount,
                    'creditor' => 0,
                ];
            }
        }

        // 6. تحويل العملة إذا العملة ليست الافتراضية
        $default_currency_id = \App\Models\General\Currency::where('is_default', 1)->value('id');
        $exchange_rate = 1;
        if ($currency_id != $default_currency_id) {
            $exchange_rate = (float) \App\Models\General\Currency::find($currency_id)->exchange_rate;
            foreach ($transfers as &$t) {
                if ($t['debit'] > 0) {
                    $t['debit'] = round($t['debit'] * $exchange_rate, 2);
                    $t['currency_debit'] = $t['debit'];
                }
                if ($t['creditor'] > 0) {
                    $t['creditor'] = round($t['creditor'] * $exchange_rate, 2);
                    $t['currency_creditor'] = $t['creditor'];
                }
            }
            unset($t);
        }

        // 7. فحص التوازن المحاسبي
        $total_debit = array_sum(array_column($transfers, 'debit'));
        $total_creditor = array_sum(array_column($transfers, 'creditor'));
        $currency_creditor = array_sum(array_column($transfers, 'currency_creditor'));
        $currency_debit = array_sum(array_column($transfers, 'currency_debit'));
        if (abs($total_debit - $total_creditor) > 0.01) {
            $this->setError(['balance' => 'عدد المدين لا يساوي الدائن']);

            return false;
        }
        // 8. تجهيز البيانات بصيغتك ثم الارسال
        $const = [
            'customer_id' => $customer_id,
            'date' => $date,
            'total_creditor' => $total_creditor,
            'total_debit' => $total_debit,
            'currency_creditor' => $currency_creditor,
            'currency_debit' => $currency_debit,
            'payType_id' => $pay_type_id,
            'type_optional' => 'wallet',
            'name' => 'wallet',
            'description' => $description,
            'active' => 1,
            'capture_exchange' => $type,
            'currency_id' => $currency_id,
            'currency_transfer_rate' => $exchange_rate,
            'company_id' => $company_id,
            'transfers' => $transfers,
        ];

        // 9. التنفيذ
        return app(\App\sys\Repository\Accounting\ConstraintRepository::class)->create($const);
    }

    /**
     * استرجاع جميع الحركات الخاصة بمحفظة العميل
     */
    public function getCustomerWalletTransactions($data)
    {

        $rules = [
            'customer_id' => ['required', 'integer', 'exists:pr_customer,id'],
            'from' => ['sometimes', 'nullable', 'date'],
            'to' => ['sometimes', 'nullable', 'date'],
            'currency_id' => ['sometimes', 'nullable',  'exists:currencies,id'],
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerProfilesRepository->getCustomerWalletTransactions($data);
    }

    /**
     * احضار الرصيد فقط لمحفظة العميل
     */
    public function getCustomerWalletBalance($data)
    {
        $rules = [
            'customer_id' => ['required', 'integer', 'exists:pr_customer,id'],
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerProfilesRepository->getCustomerWalletBalance($data);
    }

    /**
     * Summary of customers' wallets by customer
     * Params: customer_ids (array|null), from (date|null), to (date|null)
     */
    public function getCustomersWalletSummary(array $data)
    {
        $rules = [
            'customer_ids' => ['nullable', 'array'],
            'customer_ids.*' => ['integer', 'exists:pr_customer,id'],
            'from' => ['nullable', 'date'],
            'to' => ['nullable', 'date'],
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerProfilesRepository->getCustomersWalletSummary($data);
    }
}
