<?php

namespace App\sys\Services\Profile;

use App\sys\Enums\CompanyAccountMapping;
use App\sys\Enums\SupplierAccountMappings;
use App\sys\Helper;
use App\sys\Repository\Accounting\ConstraintRepository;
use App\sys\Repository\Profile\TravelerRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class TravelerService extends Services
{
    protected TravelerRepository $travelerRepository;

    public function __construct(TravelerRepository $travelerRepository)
    {
        $this->travelerRepository = $travelerRepository;
    }

    public function getPaginated($filters = [])
    {
        return $this->travelerRepository->getPaginated($filters);
    }

    public function getById($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->findByIdOrFail($id);
    }

    public function delete($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile_travelers,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->travelerRepository->findByIdOrFail($id);

        return $this->travelerRepository->delete($item);
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_profile_travelers,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->del($ids);
    }

    public function createMultiple($request)
    {
        $request = $this->normalizeTravelersRequest($request);
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'travelers' => ['required', 'array', 'min:1'],
            'travelers.*.nationality_id' => ['required', 'integer', 'exists:nationalities,id'],
            'travelers.*.currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'travelers.*.type' => ['required', 'in:adult,child'],
            'travelers.*.unit_price' => ['required', 'numeric', 'min:0'],
            'travelers.*.count' => ['required', 'integer', 'min:0'],
            'travelers.*.tax_id' => ['sometimes', 'array'],
            'travelers.*.tax_id.*' => ['integer', 'exists:taxs_rate,id'],
            'tax_id' => ['sometimes', 'array'],
            'tax_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->createMultiple($request);
        // $this->constraints($data);

    }

    private function constraints($travelers)
    {

        $types = array_column(SupplierAccountMappings::cases(), 'value');
        $accounting = $suppliers->accounting()->whereIn('type', $types)->get();
        $treeCurrnecy = $accounting->pluck('currency_id')->toArray();
        $tree = Helper::getCompanyTreeAccount($suppliers->company_id, CompanyAccountMapping::OPENING_BALANCE->value, $treeCurrnecy);

        if (! empty($tree) && ! empty($accounting)) {
            $constRepo = new ConstraintRepository;
            foreach ($accounting as $account) {
                if ($tree[(int) $account->currency_id]) {
                    $currnecy = $account->currency;
                    $rate = (float) $currnecy->is_default == 1 ? 1 : $currnecy->exchange_rate;
                    $transfers = [
                        [
                            'tree_accounting_id' => $account->tree_account_id,
                            'currency_id' => $account->currency_id,
                            'name' => 'opening_balance_supplier_'.$suppliers->supplier_name,
                            'debit' => $account->account_type == 'debit' ? round($account->balance * $rate, 2) : 0,
                            'creditor' => $account->account_type == 'credit' ? round($account->balance * $rate, 2) : 0,
                            'currency_debit' => $account->account_type == 'debit' ? round($account->balance, 2) : 0,
                            'currency_creditor' => $account->account_type == 'credit' ? round($account->balance, 2) : 0,
                            'cost_center_id' => null,
                        ],
                        [
                            'tree_accounting_id' => $tree[(int) $account->currency_id],
                            'currency_id' => $account->currency_id,
                            'name' => 'opening_balance_company_'.$suppliers->supplier_name,
                            'debit' => $account->account_type == 'credit' ? round($account->balance * $rate, 2) : 0,
                            'creditor' => $account->account_type == 'debit' ? round($account->balance * $rate, 2) : 0,
                            'currency_debit' => $account->account_type == 'credit' ? round($account->balance, 2) : 0,
                            'currency_creditor' => $account->account_type == 'debit' ? round($account->balance, 2) : 0,
                            'cost_center_id' => null,
                        ],
                    ];
                    $total_debit = array_sum(array_column($transfers, 'debit'));
                    $total_creditor = array_sum(array_column($transfers, 'creditor'));
                    $const = [
                        'supplier_id' => $suppliers->id,
                        'date' => date('Y-m-d'),
                        'total_creditor' => $total_creditor,
                        'total_debit' => $total_debit,

                        'type_optional' => 'opening_balance',
                        'name' => 'opening_balance',
                        'description' => 'opening_balance_'.$suppliers->supplier_name,
                        'active' => 1,
                        'capture_exchange' => 'constraint',
                        'creation_mode' => 'automatic',
                        'currency_id' => $account->currency_id,
                        'currency_transfer_rate' => $currnecy->exchange_rate,
                        'company_id' => $suppliers->company_id,
                        'transfers' => $transfers,

                    ];
                    $constRepo->create($const, 'yes');
                }

            }
        }
    }

    public function updateMultiple($request)
    {
        $request = $this->normalizeTravelersRequest($request);
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'travelers' => ['required', 'array', 'min:1'],
            'travelers.*.nationality_id' => ['required', 'integer', 'exists:nationalities,id'],
            'travelers.*.currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'travelers.*.type' => ['required', 'in:adult,child'],
            'travelers.*.unit_price' => ['required', 'numeric', 'min:0'],
            'travelers.*.count' => ['required', 'integer', 'min:0'],
            'travelers.*.tax_id' => ['sometimes', 'array'],
            'travelers.*.tax_id.*' => ['integer', 'exists:taxs_rate,id'],
            'tax_id' => ['sometimes', 'array'],
            'tax_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->updateMultiple($request);
    }

    private function normalizeTravelersRequest($request)
    {
        $travelers = [];
        if (isset($request['travelers']['adults']) && is_array($request['travelers']['adults'])) {
            foreach ($request['travelers']['adults'] as $traveler) {
                $traveler['type'] = 'adult';
                $travelers[] = $traveler;
            }
        }
        if (isset($request['travelers']['children']) && is_array($request['travelers']['children'])) {
            foreach ($request['travelers']['children'] as $traveler) {
                $traveler['type'] = 'child';
                $travelers[] = $traveler;
            }
        }
        $request['travelers'] = $travelers;

        return $request;
    }

    public function getAllChanges($profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->getAllChanges($profileId);
    }

    public function getTravelerNationalitiesByProfile($profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];

        $validator = Validator::make(['profile_id' => $profileId], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->getTravelerNationalities($profileId);
    }
}
