<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\ConstraintCollection;
use App\Http\Resources\Accounting\ConstraintResource;
use App\Models\Languages\Languages;
use App\sys\ApiResponse;
use App\sys\Services\Accounting\ConstraintService;
use Illuminate\Http\Request;
use PDF;

class ConstraintController extends Controller
{
    use ApiResponse;

    protected ConstraintService $service;

    public function __construct(ConstraintService $service)
    {
        $this->service = $service;
    }

    public function index()
    {
        $constraints = $this->service->getPaginated();

        return $this->apiResponse(200, 'success', null, new ConstraintCollection($constraints));
    }

    public function show($id)
    {
        $item = $this->service->getById((int) $id);
        if (! $item) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }
        $item->load(['transfers', 'attachments']);

        return $this->apiResponse(200, 'success', null, new ConstraintResource($item));
    }

    public function getConstriantByid($constraint_id)
    {

        $result = $this->service->getConstriantByid($constraint_id);

        if (! $result['success']) {
            return response()->json([
                'status' => false,
                'errors' => $result['errors'],
            ], 422);
        }

        return response()->json([
            'status' => true,
            'data' => $result['data'],
        ], 200);
    }

    public function store(Request $request)
    {
        $response = $this->service->create($request->all());

        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ConstraintResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->service->update($request->all());
        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ConstraintResource($response));
    }

    public function destroy(Request $request)
    {
        $response = $this->service->del($request->ids);
        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', 'Constraint deleted successfully', null);
    }

    public function deleteAttachments(Request $request)
    {
        $result = $this->service->deleteAttachments($request->all());
        if (! $result) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getNextDocumentNumber($type)
    {
        $number = $this->service->getNextDocumentNumber($type);

        return $this->apiResponse(200, 'success', null, ['next_document_number' => $number]);
    }

    public function activate($id)
    {
        $updated = $this->service->activate((int) $id);
        if (! $updated) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ConstraintResource($updated));
    }

    public function print($constraint_id)
    {

        if ($id = request()->query('x-lang')) {
            $language = Languages::find($id) ?? Languages::where('is_active', 1)->first();

            if ($language) {

                app()->setLocale($language->code ?? 'en');

                config(['pdf_custom.direction' => $language->rtl ? 'rtl' : 'ltr']);
            }
        }

        // ============================================================
        // 2. ⭐️ جلب البيانات وإنشاء الـ PDF
        // ============================================================

        if ($data = $this->service->print($constraint_id)) {

            $data = $data['data'];

            // نقرأ الاتجاه (سواء اتظبط من الميدليوير أو من الكود اللي فوق)
            $direction = config('pdf_custom.direction', 'ltr');
            $data['direction'] = $direction;

            // تحميل الـ PDF
            $pdf = PDF::loadView('pdf.voucher', $data, [], array_merge(
                config('pdf_custom.mpdf'),
                ['directionality' => $direction]
            ));

            // اسم الملف (باللغة الحالية)
            $currentLang = app()->getLocale();
            $fileName = "voucher_{$constraint_id}_{$currentLang}.pdf";

            $viewType = request()->query('view');

            switch ($viewType) {
                case 'html':
                    return view('pdf.voucher', $data);
                case 'download':
                    return $pdf->download($fileName);

                case 'pdf':
                default:

                    return $pdf->stream($fileName);
            }
        }

        return $this->apiResponse(400, 'false', $this->service->errors(), null);
    }

    public function printDailyConstraint($constraint_id)
    {
        if ($id = request()->query('x-lang')) {
            $language = Languages::find($id) ?? Languages::where('is_active', 1)->first();

            if ($language) {
                app()->setLocale($language->code ?? 'en');
                config(['pdf_custom.direction' => $language->rtl ? 'rtl' : 'ltr']);
            }
        }

        if ($data = $this->service->printDailyConstraint($constraint_id)) {
            $data = $data['data'];

            // Determine fields based on language
            $isRtl = config('pdf_custom.direction') === 'rtl';
            $data['description_field'] = $isRtl ? 'description' : 'description_en';
            $data['account_name_field'] = $isRtl ? 'account_name_ar' : 'account_name_en';
            $data['cost_center_field'] = $isRtl ? 'cost_center_ar' : 'cost_center_en';
            $data['direction'] = config('pdf_custom.direction', 'ltr');

            // Load PDF
            $pdf = PDF::loadView('pdf.daily_constraints.daily_constraints', $data, [], array_merge(
                config('pdf_custom.mpdf'),
                ['directionality' => $data['direction']]
            ));

            $currentLang = app()->getLocale();
            $fileName = "daily_constraint_{$constraint_id}_{$currentLang}.pdf";

            $viewType = request()->query('view');

            switch ($viewType) {
                case 'html':
                    return view('pdf.daily_constraints.daily_constraints', $data);
                case 'download':
                    return $pdf->download($fileName);
                case 'pdf':
                default:
                    return $pdf->stream($fileName);
            }
        }

        return $this->apiResponse(400, 'false', $this->service->errors(), null);
    }
}
