<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\CityCollection;
use App\Http\Resources\CityResource;
use App\Http\Resources\General\CityHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\CityService;
use Illuminate\Http\Request;

class CityController extends Controller
{
    use ApiResponse;

    protected CityService $cityService;

    private $cityTranslationService;

    public function __construct(CityService $cityService)
    {
        $this->cityService = $cityService;
        $this->cityTranslationService = new \App\sys\Services\General\CityTranslationService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $cities = $this->cityService->getPaginatedCities();

        return $this->apiResponse(200, 'success', null, new CityCollection($cities));
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $city = $this->cityService->getCityById($id);

        if (! $city) {
            $errors = $this->cityService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CityResource($city));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = $this->cityService->createCity($request->all());

        if (! $response) {
            $errors = $this->cityService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CityResource($response));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        $response = $this->cityService->updateCity($request->all());

        if (! $response) {
            $errors = $this->cityService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new CityResource($response));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->cityService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Cities deleted successfully', null, null);
        }
        $errors = $this->cityService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->cityService->getActive();
        $collect = new \App\Http\Resources\CityCollection($data);

        return $this->apiResponse(200, 'Active Cities retrieved successfully', null, $collect);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->cityTranslationService->save($request->all(), $id)) {
            $errors = $this->cityTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->cityService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\City\CityLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->cityService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->cityTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->cityTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $city = $this->cityService->getCityById($id);
        if (! $city) {
            $errors = $this->cityService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->cityService->getAllChanges($city);

        return $this->apiResponse(200, 'success', null, CityHistoryResource::collection($changes));
    }
}
