<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\DomainCollection;
use App\Http\Resources\DomainResource;
use App\Http\Resources\General\Domain\DomainLangResource;
use App\Http\Resources\General\DomainHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\DomainService;
use App\sys\Services\General\DomainTranslationService;
use Illuminate\Http\Request;

class DomainController extends Controller
{
    use ApiResponse;

    protected DomainService $domainService;

    protected DomainTranslationService $domainTranslationService;

    public function __construct(DomainService $domainService)
    {
        $this->domainService = $domainService;
        $this->domainTranslationService = new DomainTranslationService;
    }

    public function index()
    {
        $domains = $this->domainService->getPaginatedDomains();

        return $this->apiResponse(200, 'success', null, new DomainCollection($domains));
    }

    public function show(int $id)
    {
        $domain = $this->domainService->getDomainById($id);
        if (! $domain) {
            $errors = $this->domainService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new DomainResource($domain));
    }

    public function store(Request $request)
    {
        $response = $this->domainService->createDomain($request->all());
        if (! $response) {
            $errors = $this->domainService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new DomainResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->domainService->updateDomain($request->all());
        if (! $response) {
            $errors = $this->domainService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new DomainResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->domainService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Domains deleted successfully', null, null);
        }
        $errors = $this->domainService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->domainService->getActive();
        $collect = new DomainCollection($data);

        return $this->apiResponse(200, 'Active Domains retrieved successfully', null, $collect);
    }

    public function getTranslation($id)
    {
        if ($data = $this->domainService->getByIdWithTranslation($id)) {
            $collection = new DomainLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->domainService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->domainTranslationService->save($request->all(), $id)) {
            $errors = $this->domainTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function delLang($id)
    {
        if ($this->domainTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->domainTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getByTaxRateId($taxRateId)
    {
        $domains = $this->domainService->getByTaxRateId((int) $taxRateId);
        if (! $domains) {
            $errors = $this->domainService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new DomainCollection($domains));
    }

    public function getAllChanges($id)
    {
        $domain = $this->domainService->getDomainById($id);
        if (! $domain) {
            $errors = $this->domainService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->domainService->getAllChanges($domain);

        return $this->apiResponse(200, 'success', null, DomainHistoryResource::collection($changes));
    }
}
