<?php

namespace App\Observers;

use App\Models\Accounting\Invoice;
use App\sys\Enums\GroupEnum;
use App\sys\Services\Notification\NotificationService;

class InvoiceObserver
{
    protected NotificationService $notificationService;

    protected const TARGET_GROUPS = [
        GroupEnum::ADMIN,
        GroupEnum::ACCOUNTANT_MANAGER,
    ];

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    public function created(Invoice $invoice): void
    {
        $invoice->loadMissing(['customer', 'profile']);

        $serial = $invoice->serial_num ?? $invoice->id;
        $customerName = $invoice->customer?->full_name ?? 'غير محدد';
        $profileNumber = $invoice->profile?->profile_number;

        $isDeferred = (float) ($invoice->invoice_price ?? 0) == 0.0;

        $title = $isDeferred ? 'تم إنشاء فاتورة آجل' : 'تم إنشاء فاتورة جديدة';

        $messageParts = [];
        $messageParts[] = "تم إنشاء فاتورة رقم {$serial}";
        if ($profileNumber) {
            $messageParts[] = "Profile رقم {$profileNumber}";
        }
        $messageParts[] = "للعميل {$customerName}";
        if ($isDeferred) {
            $messageParts[] = 'فاتورة آجل';
        } else {
            $messageParts[] = 'فاتورة نقدي';
        }

        $message = implode(' - ', $messageParts);

        $data = [
            'invoice_id' => $invoice->id,
            'serial_num' => $invoice->serial_num,
            'profile_id' => $invoice->profile_id,
            'customer_id' => $invoice->customer_id,
            'customer_name' => $customerName,
            'invoice_price' => (float) ($invoice->invoice_price ?? 0),
            'paid_price' => (float) ($invoice->paid_price ?? 0),
            'remaining_price' => (float) ($invoice->remaining_price ?? 0),
            'action' => $isDeferred ? 'created_deferred' : 'created',
        ];

        $this->sendToGroups($invoice, $title, $message, $data);
    }

    public function updated(Invoice $invoice): void
    {
        $invoice->loadMissing(['customer', 'profile']);

        $serial = $invoice->serial_num ?? $invoice->id;
        $customerName = $invoice->customer?->full_name ?? 'غير محدد';
        $profileNumber = $invoice->profile?->profile_number;

        $paid = (float) ($invoice->paid_price ?? 0);
        $remaining = (float) ($invoice->remaining_price ?? 0);

        $title = 'تم تحديث الفاتورة';

        $messageParts = [];
        $messageParts[] = "تم تحديث الفاتورة رقم {$serial}";
        if ($profileNumber) {
            $messageParts[] = "Profile رقم {$profileNumber}";
        }
        $messageParts[] = "للعميل {$customerName}";
        $messageParts[] = "المدفوع: {$paid}";
        $messageParts[] = "المتبقي: {$remaining}";

        $message = implode(' - ', $messageParts);

        $data = [
            'invoice_id' => $invoice->id,
            'serial_num' => $invoice->serial_num,
            'profile_id' => $invoice->profile_id,
            'customer_id' => $invoice->customer_id,
            'customer_name' => $customerName,
            'paid_price' => $paid,
            'remaining_price' => $remaining,
            'old_paid_price' => (float) ($invoice->getOriginal('paid_price') ?? 0),
            'old_remaining_price' => (float) ($invoice->getOriginal('remaining_price') ?? 0),
            'action' => 'updated',
        ];

        $this->sendToGroups($invoice, $title, $message, $data);
    }

    protected function sendToGroups(Invoice $invoice, string $title, string $message, array $data): void
    {
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'invoice_update',
                title: $title,
                message: $message,
                modelName: 'Invoice',
                modelId: $invoice->id,
                data: $data,
                pushChannel: 'group'
            );
        }
    }
}
