<?php

namespace App\sys\Repository\Accommodation;

use App\Models\Accommodation\Extra;

class ExtraRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return Extra::when($name, function ($q, $name) {
            $q->whereHas('translations', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%")->where('lang_id', app('lang_id'));
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ac_extras_translation', function ($join) {
                    $join->on('ac_extras.id', '=', 'ac_extras_translation.extras_id')
                        ->where('ac_extras_translation.lang_id', app('lang_id'));
                })->orderBy('ac_extras_translation.name', $order)->select('ac_extras.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function getActive()
    {
        return Extra::with('currentTranslation')->where('is_active', 1)
            ->orderBy('updated_at', 'asc')
            ->get();
    }

    public function findByIdOrFail(int $id)
    {
        return Extra::find($id);
    }

    public function create(array $data)
    {
        $extra = new Extra;
        $extra->name = $data['name'];
        $extra->is_active = $data['active'];
        $extra->save();

        return $extra;
    }

    public function update(Extra $extra, array $data)
    {
        $extra->name = $data['name'] ?? $extra->name;
        $extra->is_active = $data['active'] ?? $extra->is_active;
        $extra->save();

        return $extra;
    }

    public function delete(Extra $extra)
    {
        return $extra->delete();
    }

    public function del(array $ids)
    {
        return \App\Models\Accommodation\Extra::whereIn('id', $ids)->delete();
    }

    public function getByIdWithTranslation($id)
    {
        return Extra::with('translations')->find($id);
    }

    public function getAllChanges($extra)
    {
        return $extra->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $audit->old_values ?? [],
                    'new_values' => $audit->new_values ?? [],
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
