<?php

namespace App\sys\Repository\Accommodation;

use App\Models\Accommodation\RoomTypes;

class RoomTypesRepository
{
    private $roomTypes;

    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function __construct()
    {
        $this->roomTypes = new RoomTypes;
    }

    public function add($data)
    {
        $this->roomTypes->name = $data['name'];
        $this->roomTypes->occupancy = $data['occupancy'];
        $this->roomTypes->is_active = $data['active'];
        $this->roomTypes->save();

        return $this->roomTypes;
    }

    public function updated($data)
    {
        $roomTypes = $this->roomTypes->find($data['id']);
        $roomTypes->name = $data['name'] ?? $roomTypes->name;
        $roomTypes->occupancy = $data['occupancy'] ?? $roomTypes->occupancy;
        $roomTypes->is_active = $data['active'] ?? $roomTypes->is_active;

        return $roomTypes;
    }

    public function get()
    {
        $name = request('name', null);
        $limit = request('limit', 15);
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');

        return $this->roomTypes->when($name, function ($q) use ($name) {
            $q->where('name', 'LIKE', "%$name%");
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ac_room_type_translation', function ($join) {
                    $join->on('ac_room_types.id', '=', 'ac_room_type_translation.room_type_id')
                        ->where('ac_room_type_translation.lang_id', app('lang_id'));
                })->orderBy('ac_room_type_translation.name', $order)->select('ac_room_types.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->orderBy('modified', 'desc')->paginate($limit);
    }

    public function getActive()
    {
        return $this->roomTypes->with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return $this->roomTypes->with('translations')->find($id);
    }

    public function getbyId($id)
    {
        return $this->roomTypes->find($id);
    }

    public function del(array $ids)
    {
        return \App\Models\Accommodation\RoomTypes::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($roomType)
    {
        return $roomType->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $audit->old_values ?? [],
                    'new_values' => $audit->new_values ?? [],
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
