<?php

namespace App\sys\Repository\Notification;

use App\Models\User;
use Illuminate\Notifications\DatabaseNotification;

class NotificationRepository
{
    /**
     * Get all notifications for a user
     */
    public function getUserNotifications(int $userId, int $limit = 10, bool $unreadOnly = false)
    {

        $query = DatabaseNotification::where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->orderBy('created_at', 'desc');
        if ($unreadOnly) {
            $query->whereNull('read_at');
        }

        return $query->paginate($limit);
    }

    /**
     * Get unread notifications count
     */
    public function getUnreadCount(int $userId): int
    {
        return DatabaseNotification::where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->whereNull('read_at')
            ->where(function ($q) {
                $q->whereNull('expires_at')
                    ->orWhere('expires_at', '>', now());
            })
            ->count();
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(string $notificationId, int $userId): bool
    {
        $notification = DatabaseNotification::where('id', $notificationId)
            ->where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->first();

        if (! $notification) {
            return false;
        }

        if (is_null($notification->read_at)) {
            $notification->markAsRead();

            return true;
        }

        return false;
    }

    /**
     * Mark all notifications as read for a user
     */
    public function markAllAsRead(int $userId): int
    {
        return DatabaseNotification::where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);
    }

    /**
     * Delete a notification
     */
    public function delete(string $notificationId, int $userId): bool
    {
        $notification = DatabaseNotification::where('id', $notificationId)
            ->where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->first();

        if (! $notification) {
            return false;
        }

        return $notification->delete();
    }

    /**
     * Get a single notification
     */
    public function find(string $notificationId, int $userId)
    {
        return DatabaseNotification::where('id', $notificationId)
            ->where('notifiable_type', 'user')
            ->where('notifiable_id', $userId)
            ->first();
    }
}
