<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\OperationDataRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class OperationDataService extends Services
{
    protected OperationDataRepository $operationDataRepository;

    public function __construct(OperationDataRepository $operationDataRepository)
    {
        $this->operationDataRepository = $operationDataRepository;
    }

    public function getPaginatedOperationData()
    {
        return $this->operationDataRepository->getPaginated();
    }

    public function getOperationDataById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_operation_data,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->operationDataRepository->findByIdOrFail($id);
    }

    public function createOperationData($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $operationData = $this->operationDataRepository->create($request);
        $this->syncTranslations($operationData);

        return $operationData;
    }

    public function updateOperationData($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_operation_data,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->operationDataRepository->findByIdOrFail($request['id']);

        return $this->operationDataRepository->update($item, $request);
    }

    public function deleteOperationData(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_operation_data,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->operationDataRepository->findByIdOrFail($id);

        return $this->operationDataRepository->delete($item);
    }

    public function del(array $ids)
    {
        return $this->operationDataRepository->del($ids);
    }

    public function getActive()
    {
        return $this->operationDataRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_operation_data,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $operationData = $this->operationDataRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($operationData, true);
    }

    public function getAllChanges($operationData)
    {
        return $this->operationDataRepository->getAllChanges($operationData);
    }
}
