<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accounting\InvoiceCollection;
use App\Http\Resources\Accounting\InvoiceResource;
use App\sys\ApiResponse;
use App\sys\Services\Accounting\InvoiceService;
use App\sys\Services\Invoice\InvoiceServicesService;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    use ApiResponse;

    protected InvoiceService $service;

    protected InvoiceServicesService $invoiceServicesService;

    public function __construct(InvoiceService $service, InvoiceServicesService $invoiceServicesService)
    {
        $this->service = $service;
        $this->invoiceServicesService = $invoiceServicesService;
    }

    public function index()
    {
        $invoices = $this->service->getPaginated();

        return $this->apiResponse(200, 'success', null, new InvoiceCollection($invoices));
    }

    public function show($id)
    {
        $item = $this->service->getById((int) $id);
        if (! $item) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceResource($item));
    }

    /**
     * Show invoice with its related services.
     * - For all services, returns basic information and service name.
     * - For accommodation services, also returns accommodation name from reservation_id.
     */
    public function showWithServices($id)
    {
        $invoice = $this->service->getById((int) $id);
        if (! $invoice) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $services = $this->invoiceServicesService
            ->getByInvoice($invoice->id)
            ->map(function ($service) {
                $serviceName = $service->service->currentTranslation->title
                    ?? $service->service->title
                    ?? null;

                $accommodationName = null;
                if ($service->travel_tourism_type === 'accommodation'
                    && $service->reservation
                    && $service->reservation->accommodation) {
                    $accommodation = $service->reservation->accommodation;
                    $accommodationName = $accommodation->currentTranslation->name
                        ?? $accommodation->name
                        ?? null;
                }

                return [
                    'id' => $service->id,
                    'travel_tourism_type' => $service->travel_tourism_type,
                    'service_id' => $service->service_id,
                    'service_name' => $serviceName,
                    'profile_id' => $service->profile_id,
                    'notes' => $service->notes,
                    'reservation_id' => $service->reservation_id,
                    'accommodation_name' => $accommodationName,
                ];
            })
            ->values();

        return $this->apiResponse(200, 'success', null,
            ['services' => $services]);
    }

    public function getByProfile($profileId)
    {
        $items = $this->service->getByProfileId((int) $profileId);
        if ($items === false) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, InvoiceResource::collection($items));
    }

    public function getByNextDate(Request $request)
    {
        $nextDate = $request->input('next_date');
        if (! $nextDate) {
            return $this->apiResponse(400, 'fail', ['next_date' => ['next_date is required']], null);
        }

        $items = $this->service->getByNextDate($nextDate);
        if ($items === false) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, InvoiceResource::collection($items));
    }

    public function store(Request $request)
    {
        $response = $this->service->create($request->all());
        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->service->update($request->all());
        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceResource($response));
    }

    public function destroy(Request $request)
    {
        $response = $this->service->del($request->ids);
        if (! $response) {
            $errors = $this->service->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', 'Invoice deleted successfully', null);
    }
}
