<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\ProfileTravelersResource;
use App\Http\Resources\Profile\TravelerCollection;
use App\Http\Resources\Profile\TravelerHistoryResource;
use App\Http\Resources\Profile\TravelerResource;
use App\sys\ApiResponse;
use App\sys\Services\Profile\TravelerService;
use Illuminate\Http\Request;

class TravelerController extends Controller
{
    use ApiResponse;

    protected TravelerService $travelerService;

    public function __construct(TravelerService $travelerService)
    {
        $this->travelerService = $travelerService;
    }

    public function index(Request $request)
    {
        $items = $this->travelerService->getPaginated($request->all());

        return $this->apiResponse(200, 'success', null, new TravelerCollection($items));
    }

    public function show($id)
    {
        $items = $this->travelerService->getById($id);
        if (! $items) {
            $errors = $this->travelerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ProfileTravelersResource($id, $items));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->travelerService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Travelers deleted successfully', null, null);
        }
        $errors = $this->travelerService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function createMultiple(Request $request)
    {
        $response = $this->travelerService->createMultiple($request->all());
        if (! $response) {
            $errors = $this->travelerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, TravelerResource::collection($response));
    }

    public function updateMultiple(Request $request)
    {
        $response = $this->travelerService->updateMultiple($request->all());
        if (! $response) {
            $errors = $this->travelerService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, TravelerResource::collection($response));
    }

    public function getAllChanges($id)
    {
        $allChanges = $this->travelerService->getAllChanges($id);
        if (! $allChanges) {
            $errors = $this->travelerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, TravelerHistoryResource::collection($allChanges));
    }

    public function getTravelerNationalitiesByProfile($profileId)
    {
        $travelers = $this->travelerService->getTravelerNationalitiesByProfile($profileId);

        if (! $travelers) {
            $errors = $this->travelerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $travelers);
    }
}
