<?php

namespace App\Http\Controllers\Roles_Groups;

use App\Http\Controllers\Controller;
use App\Http\Resources\Roles_Groups\PermissionCollection;
use App\Http\Resources\Roles_Groups\PermissionResource;
use App\sys\ApiResponse;
use App\sys\Services\Roles_Groups\PermissionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PermissionController extends Controller
{
    use ApiResponse;

    protected $permissionService;

    public function __construct(PermissionService $permissionService)
    {

        $this->permissionService = $permissionService;

    }

    public function getUniqueControllerNames()
    {
        $allPermissions = DB::table('permissions')
            ->select('id', 'permission_name', 'Controller_name', 'collection_name')
            ->orderBy('Controller_name')
            ->get();
        $formattedPermissions = $allPermissions
            ->groupBy('Controller_name')
            ->values()
            ->map(function ($permissions, $index) {
                $controllerName = $permissions->first()->Controller_name;

                return [
                    'id' => $index + 1,
                    'name' => $controllerName,
                    'key' => strtoupper(str_replace(' ', '_', $controllerName)),
                    'permissions' => $permissions->map(function ($permission) {
                        return [
                            'id' => $permission->id,
                            'permission_name' => $permission->permission_name,
                            'controller_name' => $permission->Controller_name, 'collection_name' => $permission->collection_name,
                        ];
                    }),
                ];
            });

        return $this->apiResponse(200, 'success', null, $formattedPermissions);
    }

    public function index()
    {
        $constraints = $this->permissionService->getPaginated();

        return $this->apiResponse(200, 'success', null, new PermissionCollection($constraints));
    }

    public function store(Request $request)
    {
        $response = $this->permissionService->create($request->all());

        if (! $response) {
            $errors = $this->permissionService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new PermissionResource($response));
    }

    /**
     * Return a single role.
     */
    public function show($id)
    {

        $role = $this->permissionService->get($id);
        if (! $role) {
            $errors = $this->permissionService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new PermissionResource($role));
    }

    public function update(Request $request, $id)
    {
        $data = $request->all();
        $response = $this->permissionService->updatee($data, $id);
        if (! $response) {
            $errors = $this->permissionService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new PermissionResource($response));
    }

    public function delete($id)
    {
        if ($this->permissionService->deletee($id)) {
            return $this->apiResponse(200, 'permission deleted successfully', null, null);
        }

        return $this->apiResponse(400, 'Permission not found', null, null);
    }

    public function getPermissionsByController(string $controllerName)
    {

        $permissions = DB::table('permissions')
            ->where('Controller_name', $controllerName)
            ->select('id', 'permission_name', 'collection_name')
            ->get();

        if ($permissions->isEmpty()) {

            return $this->apiResponse(404, 'Controller not found or has no permissions', null, []);
        }

        return $this->apiResponse(200, 'success', null, $permissions);
    }

    public function myPermissions()
    {
        $user = auth('api')->user();
        $permissions = $this->permissionService->getUserPermissionsGrouped($user);

        if ($permissions->isEmpty()) {

            return $this->apiResponse(404, 'Controller not found or has no permissions', null, []);
        }

        return $this->apiResponse(200, 'success', null, $permissions);
    }
}
