<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Notification;

class GenericNotification extends Notification implements ShouldBroadcast
{
    use Queueable;

    protected $notifiable;

    public function __construct(
        public string $title,
        public string $message,
        public string $type,
        public ?string $modelName = null,
        public ?int $modelId = null,
        public ?int $senderBy = null,
        public array $data = [],
        public ?string $channel = null,
        public ?\DateTime $expiresAt = null
    ) {}

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        $this->notifiable = $notifiable;

        return ['database', 'broadcast'];
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray($notifiable): array
    {
        return [
            'title' => $this->title,
            'message' => $this->message,
            'type' => $this->type,
            'model_name' => $this->modelName,
            'model_id' => $this->modelId,
            'sender_by' => $this->senderBy,
            'data' => $this->data,
        ];
    }

    /**
     * Get the broadcastable representation of the notification.
     */
    public function toBroadcast($notifiable): BroadcastMessage
    {
        $senderName = null;
        $messageWithSender = $this->message;

        if ($this->senderBy) {
            $sender = User::find($this->senderBy);
            $senderName = $sender?->name;

            // إضافة اسم المرسل في رسالة الـ push
            if ($senderName) {
                $messageWithSender = "{$senderName}: {$this->message}";
            }
        }

        return new BroadcastMessage([
            'title' => $this->title,
            'message' => $messageWithSender,
            'type' => $this->type,
            'model_name' => $this->modelName,
            'model_id' => $this->modelId,
            'sender_by' => $this->senderBy,
            'sender_name' => $senderName,
            'data' => $this->data,
        ]);
    }

    /**
     * Get the channels the event should broadcast on.
     */
    public function broadcastOn(): array
    {
        // Private channel for the recipient user (not the sender)
        // The channel property is only used for public channels via PublicNotificationEvent
        if (! $this->notifiable) {
            return [];
        }

        $userId = $this->notifiable->id;

        return [new \Illuminate\Broadcasting\PrivateChannel('user.'.$userId)];
    }
}
