<?php

namespace App\sys\Repository\Accommodation;

use App\Models\Accommodation\RoomViews;

class RoomViewsRepository
{
    private $room;

    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'type' => 'type',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function __construct()
    {
        $this->room = new RoomViews;
    }

    public function add($data)
    {
        $this->room->name = $data['name'];
        $this->room->type = $data['type'];
        $this->room->is_active = $data['active'];
        $this->room->save();

        return $this->room;
    }

    public function updated($data)
    {
        $room = $this->room->find($data['id']);
        $room->name = $data['name'] ?? $room->name;
        $room->type = $data['type'] ?? $room->type;
        $room->is_active = $data['active'] ?? $room->is_active;
        $room->save();

        return $room;
    }

    public function get()
    {
        $name = request('name', null);
        $limit = request('limit', 15);
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $type = request('type', null);

        return $this->room->when($name, function ($q, $name) {
            $q->whereHas('translations', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%")->where('lang_id', app('lang_id'));
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ac_room_views_translation', function ($join) {
                    $join->on('ac_room_views.id', '=', 'ac_room_views_translation.room_view_id')
                        ->where('ac_room_views_translation.lang_id', app('lang_id'));
                })->orderBy('ac_room_views_translation.name', $order)->select('ac_room_views.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->orderBy('updated_at', 'desc')->paginate($limit);
    }

    public function getbyId($id)
    {
        return $this->room->with('currentTranslation')->find($id);
    }

    public function getActive()
    {
        return $this->room->with('currentTranslation')->where('is_active', 1)->get();
    }

    public function ActiveWithType($type)
    {
        return $this->room->with('currentTranslation')->where('type', $type)->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return $this->room->with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return \App\Models\Accommodation\RoomViews::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($roomView)
    {
        return $roomView->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $audit->old_values ?? [],
                    'new_values' => $audit->new_values ?? [],
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
