<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\Invoice;

class InvoiceRepository
{
    private $columns = [
        'id' => 'id',
        'profile_id' => 'profile_id',
        'customer_id' => 'customer_id',
        'company_id' => 'company_id',
        'invoice_date' => 'invoice_date',
        'invoice_price' => 'invoice_price',
        'paid_price' => 'paid_price',
        'remaining_price' => 'remaining_price',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'desc');
        $profileId = request('profile_id', null);
        $nextDate = request('next_date', null);
        $status = request('payment_status', null);
        $limit = request('limit', 15);

        $query = Invoice::with(['profile.user', 'customer', 'company.currentTranslation'])
            ->when($profileId, function ($q) use ($profileId) {
                $q->where('profile_id', $profileId);
            })
            ->when($nextDate, function ($q) use ($nextDate) {
                $q->where('next_paid_date', $nextDate);
            })
            ->when($status && $status !== 'all', function ($q) use ($status) {
                if ($status === 'paid') {
                    // paid: invoice_price = paid_price AND remaining_price = 0
                    $q->whereColumn('invoice_price', 'paid_price')
                        ->where('remaining_price', 0);
                } elseif ($status === 'partiallyPaid') {
                    // partiallyPaid: (paid_price < invoice_price AND paid_price > 0) OR (remaining_price < invoice_price AND remaining_price > 0)
                    // But not paid (invoice_price != paid_price OR remaining_price != 0)
                    $q->where(function ($subQ) {
                        $subQ->where(function ($q) {
                            $q->whereColumn('paid_price', '<', 'invoice_price')
                                ->where('paid_price', '>', 0);
                        })->orWhere(function ($q) {
                            $q->whereColumn('remaining_price', '<', 'invoice_price')
                                ->where('remaining_price', '>', 0);
                        });
                    })->where(function ($subQ) {
                        // Exclude paid invoices
                        $subQ->whereColumn('invoice_price', '!=', 'paid_price')
                            ->orWhere('remaining_price', '!=', 0);
                    });
                } elseif ($status === 'deferred') {
                    // deferred: invoice_price = 0
                    $q->where('invoice_price', 0);
                }
            });

        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        } else {
            $query->orderBy('id', $order);
        }

        return $query->paginate($limit);
    }

    public function findById($id)
    {
        return Invoice::with(['profile.user', 'customer', 'company.currentTranslation'])->find($id);
    }

    public function getByProfileId($profileId)
    {
        return Invoice::with(['profile.user', 'customer', 'company.currentTranslation'])
            ->where('profile_id', $profileId)
            ->get();
    }

    public function getByNextDate($nextDate)
    {
        return Invoice::with(['profile.user', 'customer', 'company.currentTranslation'])
            ->where('next_paid_date', $nextDate)
            ->get();
    }

    public function getNextSerialNumber()
    {
        // Get all serial numbers that are not null or empty
        $serials = Invoice::whereNotNull('serial_num')
            ->where('serial_num', '!=', '')
            ->pluck('serial_num')
            ->filter();

        if ($serials->isEmpty()) {
            return 1;
        }

        $maxNumber = 0;

        foreach ($serials as $serial) {
            // If it's a pure number, use it directly
            if (is_numeric($serial)) {
                $maxNumber = max($maxNumber, (int) $serial);
            } else {
                // If it's a string, try to extract the last number sequence
                // Example: "INV-2024-001" -> extract "001" -> 1
                if (preg_match('/(\d+)(?!.*\d)/', $serial, $matches)) {
                    $maxNumber = max($maxNumber, (int) $matches[1]);
                }
            }
        }

        return $maxNumber + 1;
    }

    public function create(array $data)
    {
        $invoice = new Invoice;
        $invoice->serial_num = $data['serial_num'] ?? $this->getNextSerialNumber();
        $invoice->profile_id = $data['profile_id'] ?? null;
        $invoice->customer_id = $data['customer_id'] ?? null;
        $invoice->company_id = $data['company_id'] ?? null;
        $invoice->invoice_date = $data['invoice_date'] ?? now()->format('Y-m-d');
        $invoice->recip_date = $data['recip_date'] ?? $data['invoice_date'] ?? now()->format('Y-m-d');
        $invoice->pay_duration = $data['pay_duration'] ?? null;
        $invoice->invoice_date_due = $data['invoice_date_due'] ?? null;
        $invoice->invoice_price = $data['invoice_price'] ?? 0;
        $invoice->invoice_type = $data['invoice_type'] ?? 1;
        $invoice->paid_price = $data['paid_price'] ?? 0;
        $invoice->paid_date = $data['paid_date'] ?? null;
        $invoice->remaining_price = $data['remaining_price'] ?? 0;
        $invoice->worthy_days = $data['worthy_days'] ?? null;
        $invoice->notes = $data['notes'] ?? null;
        $invoice->conditions = $data['conditions'] ?? null;
        $invoice->next_paid_date = $data['next_paid_date'] ?? null;
        $invoice->save();

        return $invoice;
    }

    public function update(Invoice $invoice, array $data)
    {
        $invoice->serial_num = $data['serial_num'] ?? $invoice->serial_num;
        $invoice->profile_id = $data['profile_id'] ?? $invoice->profile_id;
        $invoice->customer_id = $data['customer_id'] ?? $invoice->customer_id;
        $invoice->company_id = $data['company_id'] ?? $invoice->company_id;
        $invoice->invoice_date = $data['invoice_date'] ?? $invoice->invoice_date;
        $invoice->recip_date = $data['recip_date'] ?? $invoice->recip_date;
        $invoice->pay_duration = $data['pay_duration'] ?? $invoice->pay_duration;
        $invoice->invoice_date_due = $data['invoice_date_due'] ?? $invoice->invoice_date_due;
        $invoice->invoice_price = $data['invoice_price'] ?? $invoice->invoice_price;
        $invoice->invoice_type = $data['invoice_type'] ?? $invoice->invoice_type;
        $invoice->paid_price = $data['paid_price'] ?? $invoice->paid_price;
        $invoice->paid_date = $data['paid_date'] ?? $invoice->paid_date;
        $invoice->remaining_price = $data['remaining_price'] ?? $invoice->remaining_price;
        $invoice->worthy_days = $data['worthy_days'] ?? $invoice->worthy_days;
        $invoice->notes = $data['notes'] ?? $invoice->notes;
        $invoice->conditions = $data['conditions'] ?? $invoice->conditions;
        $invoice->next_paid_date = $data['next_paid_date'] ?? $invoice->next_paid_date;
        $invoice->save();

        return $invoice;
    }

    public function del(array $ids)
    {
        return Invoice::whereIn('id', $ids)->delete();
    }
}
