<?php

namespace App\sys\Repository\Invoice;

use App\Models\Accounting\Constraint;
use App\Models\Accounting\Transfer;
use App\Models\General\City;
use App\Models\General\Companies;
use App\Models\General\Currency;
use App\Models\General\GuideLanguage;
use App\Models\General\Nationality;
use App\Models\General\OperationData;
use App\Models\General\Service as GeneralService;
use App\Models\General\ServicesCurrencies;
use App\Models\General\TaxRateMappings;
use App\Models\invoice\InvoiceServices;
use App\Models\invoice\InvoiceServicesTax;
use App\Models\Profile\Profile;
use App\Models\Suppliers\SupplierAccountMappings;
use App\Models\Suppliers\Suppliers;
use App\Models\User;
use App\sys\Enums\GroupEnum;
use App\sys\Enums\SupplierAccountMappings as SupplierAccountMappingsEnum;
use Illuminate\Support\Facades\DB;

class InvoiceServicesRepository
{
    public function findByIdOrFail(int $id)
    {
        return InvoiceServices::with($this->relations())->find($id);
    }

    public function findByModel(string $modelType, int $modelId)
    {
        return InvoiceServices::where('model_type', $modelType)
            ->where('model_id', $modelId)
            ->first();
    }

    public function create(array $data)
    {
        $item = new InvoiceServices;

        $item->travel_tourism_type = $data['travel_tourism_type'] ?? 'accommodation';
        $item->type = $data['type'] ?? 'credit';
        $item->created_by = Auth()->guard('api')->user()->id ?? null;

        // جلب salse_id من profile->user_id إذا لم يكن موجوداً في البيانات
        if (isset($data['profile_id']) && empty($data['salse_id'])) {
            $profile = \App\Models\Profile\Profile::find($data['profile_id']);
            $item->salse_id = $profile?->user_id ?? null;
        } else {
            $item->salse_id = $data['salse_id'] ?? null;
        }
        $item->travel_tourism_steps = $data['travel_tourism_steps'] ?? 0;
        $item->travel_tourism_done = $data['travel_tourism_done'] ?? 0;
        $item->service_id = $data['service_id'] ?? null;
        $item->domain_id = $data['domain_id'] ?? null;
        $item->invoice_id = $data['invoice_id'] ?? null;
        $item->supplier_id = $data['supplier_id'] ?? null;
        $item->category_id = $data['category_id'] ?? null;
        $item->pay_type_id = $data['pay_type_id'] ?? null;
        $item->pay_type = $data['pay_type'] ?? null;
        $item->reserve_number = $data['reserve_number'] ?? null;
        $item->vatt_amount = $data['vatt_amount'] ?? null;
        $item->vatt_Supplier = $data['vatt_Supplier'] ?? null;
        $item->profit = $data['profit'] ?? null;
        $item->employee_tax = $data['employee_tax'] ?? null;
        $item->last_profit = $data['last_profit'] ?? null;
        $item->taxtree = $data['taxtree'] ?? null;
        $item->supplirTree = $data['supplirTree'] ?? null;
        $item->serviceTree = $data['serviceTree'] ?? null;
        $item->payTree = $data['payTree'] ?? null;
        $item->serviceTreeIncom = $data['serviceTreeIncom'] ?? null;
        $item->serviceTypeAndPayType = $data['serviceTypeAndPayType'] ?? null;
        $item->paid_rice = $data['paid_rice'] ?? null;
        $item->supplier_currency_id = $data['supplier_currency_id'] ?? null;
        $item->supplier_currency_transfer_rate = $data['supplier_currency_transfer_rate'] ?? null;
        $item->client_currency_id = $data['client_currency_id'] ?? null;
        $item->client_currency_transfer_rate = $data['client_currency_transfer_rate'] ?? null;
        $item->extra_service_price = $data['extra_service_price'] ?? 0;
        $item->extra_service_tree = $data['extra_service_tree'] ?? null;

        // Attractions/Temples and similar dynamic fields
        $item->daily_program_id = $data['daily_program_id'] ?? null;
        $item->nationality_id = $data['nationality_id'] ?? null;
        $item->adults_count = $data['adults_count'] ?? 0;
        $item->children_count = $data['children_count'] ?? 0;
        $item->adult_price = $data['adult_price'] ?? 0;
        $item->child_price = $data['child_price'] ?? 0;
        $item->adult_total = $data['adult_total'] ?? 0;
        $item->child_total = $data['child_total'] ?? 0;
        // Common dynamic fields across types
        $item->people_count = $data['people_count'] ?? 0;
        $item->daily_cost = $data['daily_cost'] ?? 0;
        $item->extra_cost = $data['extra_cost'] ?? 0;
        $item->total_tax = $data['total_tax'] ?? 0;
        $item->tip_amount = $data['tip_amount'] ?? 0;
        $item->total_tips = $data['total_tips'] ?? 0;
        $item->company_id = $data['company_id'] ?? null;
        $item->city_id = $data['city_id'] ?? null;
        $item->city_to_id = $data['city_to_id'] ?? null;
        $item->operation_data_id = $data['operation_data_id'] ?? null;
        $item->guide_language_id = $data['guide_language_id'] ?? null;
        // tour_guide specifics
        $item->guide_id = $data['guide_id'] ?? null;
        $item->grand_total = $data['grand_total'] ?? 0;
        $item->execution_date = $data['execution_date'] ?? null;
        $item->executive_id = $data['executive_id'] ?? null;
        $item->currency_id = $data['currency_id'] ?? null;
        $item->currency_rate = $data['currency_rate'] ?? 0;
        $item->notes = $data['notes'] ?? $item->notes ?? null;
        $item->purchase_price = $data['purchase_price'] ?? null;
        $item->purchase_commission = $data['purchase_commission'] ?? null;
        $item->purchase_rate = $data['purchase_rate'] ?? null;
        $item->purchase_tax_id = $data['purchase_tax_id'] ?? null;
        $item->purchase_tax_amount = $data['purchase_tax_amount'] ?? null;
        $item->purchase_currency_id = $data['purchase_currency_id'] ?? null;
        $item->purchase_tax_rate = $data['purchase_tax_rate'] ?? null;
        $item->sale_price = $data['sale_price'] ?? null;
        $item->sale_commission = $data['sale_commission'] ?? null;
        $item->sale_rate = $data['sale_rate'] ?? null;
        $item->sale_tax_id = $data['sale_tax_id'] ?? null;
        $item->sale_tax_amount = $data['sale_tax_amount'] ?? null;
        $item->sale_currency_id = $data['sale_currency_id'] ?? null;
        $item->sale_tax_rate = $data['sale_tax_rate'] ?? null;
        $item->model_type = $data['model_type'] ?? null;
        $item->model_id = $data['model_id'] ?? null;
        $item->profile_id = $data['profile_id'] ?? null;
        $item->reservation_id = $data['reservation_id'] ?? null;
        $item->reservation_model = $data['reservation_model'] ?? null;
        $item->is_by_handling = $data['is_by_handling'] ?? 0;
        $item->is_paid = 0;
        $item->paid_to_supplier = 0;
        $item->remaining_to_supplier = $data['grand_total'] ?? 0;
        // Flights specific fields
        $item->pnr_number = $data['pnr_number'] ?? null;
        $item->office_id = $data['office_id'] ?? null;
        $item->ticket_date = $data['ticket_date'] ?? null;
        $item->direction_type = $data['direction_type'] ?? null;
        $item->economy_class = $data['economy_class'] ?? 'Y';
        $item->supplier_amount = $data['supplier_amount'] ?? null;

        // Flights taxes breakdown (stored as JSON text, casted to array in model)
        $item->air_taxs = $data['air_taxs'] ?? null;

        // Selling-specific fields
        $item->selling_currency_id = $data['selling_currency_id'] ?? null;
        $item->selling_price = $data['selling_price'] ?? null;
        $item->selling_total_tax = $data['selling_total_tax'] ?? null;
        $item->selling_adult_price = $data['selling_adult_price'] ?? null;
        $item->selling_child_price = $data['selling_child_price'] ?? null;
        $item->save();

        return $item->load($this->relations());
    }

    private function handleByHandlingInvoice(InvoiceServices $item)
    {
        // If not accommodation, create invoice only (without customer credit constraint)
        if ($item->travel_tourism_type !== 'accommodation') {
            $invoice = $this->createInvoiceForInvoiceService($item);

            if ($invoice) {
                // Update invoice_id for this invoice service
                $item->invoice_id = $invoice->id;
                $item->save();
            }
        } else {
            // For accommodation: create constraint only (customer credit) and link to invoice
            $this->createAccommodationConstraint($item);
        }
    }

    /**
     * Create invoice for invoice service only (without customer credit constraint)
     * This is used when is_by_handling = 1 and we don't know the price yet
     */
    private function createInvoiceForInvoiceService(InvoiceServices $item)
    {
        $profile = Profile::find($item->profile_id);
        if (! $profile) {
            return null;
        }

        // Get or create invoice for the profile
        $invoice = \App\Models\Accounting\Invoice::where('profile_id', $profile->id)->first();

        if (! $invoice) {
            // Create new invoice with zero amounts (price not known yet)
            $invoiceRepository = new \App\sys\Repository\Accounting\InvoiceRepository;
            $invoiceData = [
                'profile_id' => $profile->id,
                'customer_id' => $profile->customer_id,
                'company_id' => $profile->company_id,
                'invoice_date' => now()->format('Y-m-d'),
                'recip_date' => now()->format('Y-m-d'),
                'invoice_price' => 0,
                'paid_price' => 0,
                'remaining_price' => 0,
                'invoice_type' => 1,
            ];

            $invoice = $invoiceRepository->create($invoiceData);

            // Update profile status
            $profile->has_invoice = 1;
            $profile->status = 'followup';
            $profile->save();
        }

        return $invoice;
    }

    private function createAccommodationConstraint(InvoiceServices $item)
    {
        // Get or create invoice for the profile
        $profile = Profile::find($item->profile_id);
        if (! $profile) {
            return;
        }

        // Get existing invoice or create new one
        $invoice = \App\Models\Accounting\Invoice::where('profile_id', $profile->id)->first();

        if (! $invoice) {
            $unpostedService = new \App\sys\Services\Accounting\UnpostedCollectionsServices;
            $invoice = $unpostedService->createInvoiceWithoutPayment($profile->id);

            if (! $invoice) {
                return;
            }
        }

        // Update invoice_id for this invoice service
        $item->invoice_id = $invoice->id;
        $item->save();

        // Get currency and calculate amounts
        $currency = Currency::find($item->currency_id);
        if (! $currency) {
            return;
        }

        $defaultCurrency = Currency::where('is_default', 1)->first();
        if (! $defaultCurrency) {
            return;
        }

        $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);

        // Get amount - use purchase_price for accommodation
        $amount = (float) ($item->purchase_price ?? $item->grand_total ?? 0);
        if ($amount <= 0) {
            return;
        }

        // Get customer credit account
        $customerAccount = \App\Models\General\CompanyAccountMappings::where('company_id', $profile->company_id)
            ->where('currency_id', $item->currency_id)
            ->where('type', \App\sys\Enums\CompanyAccountMapping::CUSTOMER_CREDIT->value)
            ->first();

        if (! $customerAccount) {
            return;
        }

        // Get revenue account
        $revenueAccount = \App\Models\General\CompanyAccountMappings::where('company_id', $profile->company_id)
            ->where('currency_id', $item->currency_id)
            ->where('type', \App\sys\Enums\CompanyAccountMapping::PACKAGE_SERVICE_REVENUE->value)
            ->first();

        if (! $revenueAccount) {
            return;
        }

        // Calculate amounts in default currency
        $amountInDefault = round($amount * $exchangeRate, 2);

        // Create transfers for constraint
        $transfers = [
            [
                'tree_accounting_id' => $customerAccount->tree_account_id,
                'currency_id' => $item->currency_id,
                'name' => 'قيد آجل عملاء - خدمة إقامة رقم '.$item->id,
                'debit' => $amountInDefault,
                'creditor' => 0,
                'currency_debit' => round($amount, 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ],
            [
                'tree_accounting_id' => $revenueAccount->tree_account_id,
                'currency_id' => $item->currency_id,
                'name' => 'إيراد خدمات إقامة - خدمة رقم '.$item->id,
                'debit' => 0,
                'creditor' => $amountInDefault,
                'currency_debit' => 0,
                'currency_creditor' => round($amount, 2),
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ],
        ];

        // Create constraint
        $constraintData = [
            'profile_id' => $profile->id,
            'invoice_id' => $invoice->id,
            'date' => date('Y-m-d'),
            'total_creditor' => $amountInDefault,
            'total_debit' => $amountInDefault,
            'difference' => 0,
            'name' => 'قيد آجل عملاء - خدمة إقامة',
            'description' => 'قيد آجل عملاء لخدمة إقامة رقم '.$item->id.' - ملف '.$profile->profile_number,
            'active' => 1,
            'capture_exchange' => 'constraint',
            'creation_mode' => 'automatic',
            'currency_id' => $item->currency_id,
            'currency_transfer_rate' => $exchangeRate,
            'company_id' => $profile->company_id,
            'customer_id' => $profile->customer_id,
            'transfers' => $transfers,
        ];

        $constraintRepo = new \App\sys\Repository\Accounting\ConstraintRepository;
        $constraintRepo->create($constraintData);
    }

    /**
     * Create customer credit constraint for by-handling services using selling_price.
     */
    public function createSellingCustomerConstraint(InvoiceServices $item): bool
    {
        return $this->createSellingCustomerConstraintInternal($item);
    }

    private function createSellingCustomerConstraintInternal(InvoiceServices $item): bool
    {
        $profile = Profile::find($item->profile_id);
        if (! $profile) {
            return false;
        }

        $sellingCurrencyId = (int) ($item->selling_currency_id ?? 0);
        $amount = (float) ($item->selling_price ?? 0);
        if ($sellingCurrencyId <= 0 || $amount <= 0) {
            return false;
        }

        // Get existing invoice or create new one
        $invoice = \App\Models\Accounting\Invoice::where('profile_id', $profile->id)->first();

        if (! $invoice) {
            $currency = Currency::find($sellingCurrencyId);
            if (! $currency) {
                return false;
            }

            $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);
            $amountInDefault = round($amount * $exchangeRate, 2);

            $invoiceRepository = new \App\sys\Repository\Accounting\InvoiceRepository;
            $invoiceData = [
                'profile_id' => $profile->id,
                'customer_id' => $profile->customer_id,
                'company_id' => $profile->company_id,
                'invoice_date' => now()->format('Y-m-d'),
                'recip_date' => now()->format('Y-m-d'),
                'invoice_price' => $amountInDefault,
                'paid_price' => 0,
                'remaining_price' => $amountInDefault,
                'invoice_type' => 1,
            ];

            $invoice = $invoiceRepository->create($invoiceData);
            if (! $invoice) {
                return false;
            }

            // Update profile status
            $profile->has_invoice = 1;
            $profile->status = 'followup';
            $profile->save();
        }

        // Update invoice_id for this invoice service
        $item->invoice_id = $invoice->id;
        $item->save();

        // Get currency and calculate amounts
        $currency = Currency::find($sellingCurrencyId);
        if (! $currency) {
            return false;
        }

        $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);
        $amountInDefault = round($amount * $exchangeRate, 2);

        // Get customer credit account
        $customerAccount = \App\Models\General\CompanyAccountMappings::where('company_id', $profile->company_id)
            ->where('currency_id', $sellingCurrencyId)
            ->where('type', \App\sys\Enums\CompanyAccountMapping::CUSTOMER_CREDIT->value)
            ->first();

        if (! $customerAccount) {
            return false;
        }

        // Get revenue account
        $revenueAccount = \App\Models\General\CompanyAccountMappings::where('company_id', $profile->company_id)
            ->where('currency_id', $sellingCurrencyId)
            ->where('type', \App\sys\Enums\CompanyAccountMapping::PACKAGE_SERVICE_REVENUE->value)
            ->first();

        if (! $revenueAccount) {
            return false;
        }

        $serviceLabel = $item->travel_tourism_type ?? 'service';

        // Create transfers for constraint
        $transfers = [
            [
                'tree_accounting_id' => $customerAccount->tree_account_id,
                'currency_id' => $sellingCurrencyId,
                'name' => 'قيد آجل عملاء - خدمة '.$serviceLabel.' رقم '.$item->id,
                'debit' => $amountInDefault,
                'creditor' => 0,
                'currency_debit' => round($amount, 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ],
            [
                'tree_accounting_id' => $revenueAccount->tree_account_id,
                'currency_id' => $sellingCurrencyId,
                'name' => 'إيراد خدمات - خدمة '.$serviceLabel.' رقم '.$item->id,
                'debit' => 0,
                'creditor' => $amountInDefault,
                'currency_debit' => 0,
                'currency_creditor' => round($amount, 2),
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ],
        ];

        // Create constraint
        $constraintData = [
            'profile_id' => $profile->id,
            'invoice_id' => $invoice->id,
            'date' => date('Y-m-d'),
            'total_creditor' => $amountInDefault,
            'total_debit' => $amountInDefault,
            'difference' => 0,
            'name' => 'قيد آجل عملاء - خدمة '.$serviceLabel,
            'description' => 'قيد آجل عملاء لخدمة '.$serviceLabel.' رقم '.$item->id.' - ملف '.$profile->profile_number,
            'active' => 1,
            'capture_exchange' => 'constraint',
            'creation_mode' => 'automatic',
            'currency_id' => $sellingCurrencyId,
            'currency_transfer_rate' => $exchangeRate,
            'company_id' => $profile->company_id,
            'customer_id' => $profile->customer_id,
            'transfers' => $transfers,
        ];

        $constraintRepo = new \App\sys\Repository\Accounting\ConstraintRepository;
        $constraintRepo->create($constraintData);

        return true;
    }

    public function update(InvoiceServices $item, array $data)
    {
        $item->travel_tourism_type = $data['travel_tourism_type'] ?? $item->travel_tourism_type;
        $item->travel_tourism_steps = $data['travel_tourism_steps'] ?? $item->travel_tourism_steps;
        $item->travel_tourism_done = $data['travel_tourism_done'] ?? $item->travel_tourism_done;
        $item->service_id = $data['service_id'] ?? $item->service_id;
        $item->domain_id = $data['domain_id'] ?? $item->domain_id;
        $item->invoice_id = $data['invoice_id'] ?? $item->invoice_id;
        $item->supplier_id = $data['supplier_id'] ?? $item->supplier_id;
        $item->category_id = $data['category_id'] ?? $item->category_id;
        $item->pay_type_id = $data['pay_type_id'] ?? $item->pay_type_id;
        $item->pay_type = $data['pay_type'] ?? $item->pay_type;
        $item->reserve_number = $data['reserve_number'] ?? $item->reserve_number;
        $item->vatt_amount = $data['vatt_amount'] ?? $item->vatt_amount;
        $item->vatt_Supplier = $data['vatt_Supplier'] ?? $item->vatt_Supplier;
        $item->profit = $data['profit'] ?? $item->profit;
        $item->employee_tax = $data['employee_tax'] ?? $item->employee_tax;
        $item->last_profit = $data['last_profit'] ?? $item->last_profit;
        $item->taxtree = $data['taxtree'] ?? $item->taxtree;
        $item->supplirTree = $data['supplirTree'] ?? $item->supplirTree;
        $item->serviceTree = $data['serviceTree'] ?? $item->serviceTree;
        $item->payTree = $data['payTree'] ?? $item->payTree;
        $item->serviceTreeIncom = $data['serviceTreeIncom'] ?? $item->serviceTreeIncom;
        $item->serviceTypeAndPayType = $data['serviceTypeAndPayType'] ?? $item->serviceTypeAndPayType;
        $item->paid_rice = $data['paid_rice'] ?? $item->paid_rice;
        $item->supplier_currency_id = $data['supplier_currency_id'] ?? $item->supplier_currency_id;
        $item->supplier_currency_transfer_rate = $data['supplier_currency_transfer_rate'] ?? $item->supplier_currency_transfer_rate;
        $item->client_currency_id = $data['client_currency_id'] ?? $item->client_currency_id;
        $item->client_currency_transfer_rate = $data['client_currency_transfer_rate'] ?? $item->client_currency_transfer_rate;
        $item->extra_service_price = $data['extra_service_price'] ?? $item->extra_service_price;
        $item->extra_service_tree = $data['extra_service_tree'] ?? $item->extra_service_tree;
        $item->purchase_price = $data['purchase_price'] ?? $item->purchase_price;
        $item->purchase_commission = $data['purchase_commission'] ?? $item->purchase_commission;
        $item->purchase_rate = $data['purchase_rate'] ?? $item->purchase_rate;
        $item->purchase_tax_id = $data['purchase_tax_id'] ?? $item->purchase_tax_id;
        $item->purchase_tax_amount = $data['purchase_tax_amount'] ?? $item->purchase_tax_amount;
        $item->purchase_currency_id = $data['purchase_currency_id'] ?? $item->purchase_currency_id;
        $item->purchase_tax_rate = $data['purchase_tax_rate'] ?? $item->purchase_tax_rate;
        $item->sale_price = $data['sale_price'] ?? $item->sale_price;
        $item->sale_commission = $data['sale_commission'] ?? $item->sale_commission;
        $item->sale_rate = $data['sale_rate'] ?? $item->sale_rate;
        $item->sale_tax_id = $data['sale_tax_id'] ?? $item->sale_tax_id;
        $item->sale_tax_amount = $data['sale_tax_amount'] ?? $item->sale_tax_amount;
        $item->sale_currency_id = $data['sale_currency_id'] ?? $item->sale_currency_id;
        $item->sale_tax_rate = $data['sale_tax_rate'] ?? $item->sale_tax_rate;
        // Dynamic type fields
        $item->daily_program_id = $data['daily_program_id'] ?? $item->daily_program_id;
        $item->nationality_id = $data['nationality_id'] ?? $item->nationality_id;
        $item->adults_count = $data['adults_count'] ?? $item->adults_count;
        $item->children_count = $data['children_count'] ?? $item->children_count;
        $item->adult_price = $data['adult_price'] ?? $item->adult_price;
        $item->child_price = $data['child_price'] ?? $item->child_price;
        $item->adult_total = $data['adult_total'] ?? $item->adult_total;
        $item->child_total = $data['child_total'] ?? $item->child_total;
        // Common dynamic fields across types
        $item->people_count = $data['people_count'] ?? $item->people_count;
        $item->daily_cost = $data['daily_cost'] ?? $item->daily_cost;
        $item->extra_cost = array_key_exists('extra_cost', $data) ? $data['extra_cost'] : $item->extra_cost;
        $item->total_tax = array_key_exists('total_tax', $data) ? $data['total_tax'] : $item->total_tax;
        $item->tip_amount = $data['tip_amount'] ?? $item->tip_amount;
        $item->total_tips = $data['total_tips'] ?? $item->total_tips;
        $item->company_id = $data['company_id'] ?? $item->company_id;
        $item->city_id = $data['city_id'] ?? $item->city_id;
        $item->city_to_id = $data['city_to_id'] ?? $item->city_to_id;
        $item->operation_data_id = $data['operation_data_id'] ?? $item->operation_data_id;
        $item->guide_language_id = $data['guide_language_id'] ?? $item->guide_language_id;
        // tour_guide specifics
        $item->guide_id = array_key_exists('guide_id', $data) ? $data['guide_id'] : $item->guide_id;
        $item->grand_total = $data['grand_total'] ?? $item->grand_total;
        $item->execution_date = array_key_exists('execution_date', $data) ? $data['execution_date'] : $item->execution_date;
        $item->executive_id = $data['executive_id'] ?? $item->executive_id;
        $item->currency_id = $data['currency_id'] ?? $item->currency_id;
        $item->currency_rate = $data['currency_rate'] ?? $item->currency_rate;
        $item->notes = array_key_exists('notes', $data) ? $data['notes'] : $item->notes;
        $item->type = $data['type'] ?? $item->type;
        $item->model_type = $data['model_type'] ?? $item->model_type;
        $item->model_id = $data['model_id'] ?? $item->model_id;
        $item->profile_id = $data['profile_id'] ?? $item->profile_id;
        $item->reservation_id = $data['reservation_id'] ?? $item->reservation_id;
        $item->reservation_model = $data['reservation_model'] ?? $item->reservation_model;
        $item->is_by_handling = $data['is_by_handling'] ?? $item->is_by_handling;
        $item->paid_to_supplier = $data['paid_to_supplier'] ?? $item->paid_to_supplier;
        $item->remaining_to_supplier = $data['remaining_to_supplier'] ?? $item->remaining_to_supplier;
        $item->is_paid = $data['is_paid'] ?? $item->is_paid;
        $item->next_pay_date = $data['next_pay_date'] ?? $item->next_pay_date;
        // Flights specific fields
        $item->pnr_number = array_key_exists('pnr_number', $data) ? $data['pnr_number'] : $item->pnr_number;
        $item->office_id = array_key_exists('office_id', $data) ? $data['office_id'] : $item->office_id;
        $item->ticket_date = array_key_exists('ticket_date', $data) ? $data['ticket_date'] : $item->ticket_date;
        $item->direction_type = array_key_exists('direction_type', $data) ? $data['direction_type'] : $item->direction_type;
        $item->economy_class = array_key_exists('economy_class', $data) ? $data['economy_class'] : $item->economy_class;
        $item->supplier_amount = array_key_exists('supplier_amount', $data) ? $data['supplier_amount'] : $item->supplier_amount;

        // Flights taxes breakdown
        if (array_key_exists('air_taxs', $data)) {
            $item->air_taxs = $data['air_taxs'];
        }

        // Selling-specific fields
        $item->selling_currency_id = array_key_exists('selling_currency_id', $data) ? $data['selling_currency_id'] : $item->selling_currency_id;
        $item->selling_price = array_key_exists('selling_price', $data) ? $data['selling_price'] : $item->selling_price;
        $item->selling_total_tax = array_key_exists('selling_total_tax', $data) ? $data['selling_total_tax'] : $item->selling_total_tax;
        $item->selling_adult_price = array_key_exists('selling_adult_price', $data) ? $data['selling_adult_price'] : $item->selling_adult_price;
        $item->selling_child_price = array_key_exists('selling_child_price', $data) ? $data['selling_child_price'] : $item->selling_child_price;

        // تحديث updated_by
        $item->updated_by = Auth()->guard('api')->user()->id ?? null;

        // تحديث salse_id من profile->user_id إذا تغير profile_id أو لم يكن salse_id موجوداً
        if (isset($data['profile_id']) && (empty($item->salse_id) || $item->profile_id != $data['profile_id'])) {
            $profile = \App\Models\Profile\Profile::find($data['profile_id']);
            $item->salse_id = $profile?->user_id ?? $item->salse_id;
        } elseif (isset($data['salse_id'])) {
            $item->salse_id = $data['salse_id'];
        }

        // Note: status should only be updated through updateStatus method, not through regular update
        // $item->status = $data['status'] ?? $item->status;
        $item->save();

        return $item->load($this->relations());
    }

    /**
     * Update only status field (separate from regular update)
     */
    public function updateStatus(InvoiceServices $item, string $status)
    {
        $item->status = $status;
        $item->save();

        return $item->load($this->relations());
    }

    public function delete(InvoiceServices $item)
    {
        return $item->delete();
    }

    public function deleteManyWithRules(array $ids): array
    {
        $items = InvoiceServices::whereIn('id', $ids)->get();

        $deletable = [];
        $blocked = [];

        foreach ($items as $item) {
            $steps = (int) ($item->travel_tourism_steps ?? 0);
            $done = (int) ($item->travel_tourism_done ?? 0);
            // block delete if any of them equals 1 → allow only when both are not 1
            if ($steps != 1 && $done != 1) {
                $deletable[] = $item->id;
            } else {
                $blocked[] = [
                    'id' => $item->id,
                    'reason' => 'cannot delete: travel_tourism_steps = 1 or travel_tourism_done = 1',
                ];
            }
        }

        if (! empty($deletable)) {
            InvoiceServices::whereIn('id', $deletable)->delete();
        }

        return [
            'deleted_ids' => $deletable,
            'blocked' => $blocked,
        ];
    }

    public function getByProfile(int $profileId, string $serviceType)
    {
        $limit = request('limit', 15);

        return InvoiceServices::with($this->relations())
            ->where([['profile_id', $profileId], ['travel_tourism_type', $serviceType]])->paginate($limit);
    }

    public function getByDaily(int $daily, string $serviceType)
    {
        $limit = request('limit', 15);

        return InvoiceServices::with($this->relations())
            ->where([['daily_program_id', $daily], ['travel_tourism_type', $serviceType]])->paginate($limit);
    }

    /**
     * Get all services for a given invoice, with relations needed for displaying names.
     */
    public function getByInvoice(int $invoiceId)
    {
        return InvoiceServices::with([
            'service.currentTranslation',
            'reservation.accommodation.currentTranslation',
        ])
            ->where('invoice_id', $invoiceId)
            ->get();
    }

    public function getList(array $filters = [])
    {
        $query = InvoiceServices::with($this->relations())->query();

        if (isset($filters['profile_id'])) {
            $query->where('profile_id', $filters['profile_id']);
        }

        if (isset($filters['travel_tourism_type'])) {
            $query->where('travel_tourism_type', $filters['travel_tourism_type']);
        }

        if (isset($filters['service_id'])) {
            $query->where('service_id', $filters['service_id']);
        }

        if (isset($filters['invoice_id'])) {
            $query->where('invoice_id', $filters['invoice_id']);
        }

        // Add pagination if needed
        $limit = $filters['limit'] ?? 15;

        return $query->paginate($limit);
    }

    private function getFillableFields(): array
    {
        return [
            'travel_tourism_type',
            'travel_tourism_steps',
            'travel_tourism_done',
            'service_id',
            'domain_id',
            'invoice_id',
            'supplier_id',
            'pay_type_id',
            'pay_type',
            'reserve_number',
            'vatt_amount',
            'vatt_Supplier',
            'profit',
            'employee_tax',
            'last_profit',
            'taxtree',
            'supplirTree',
            'serviceTree',
            'payTree',
            'serviceTreeIncom',
            'serviceTypeAndPayType',
            'paid_rice',
            'supplier_currency_id',
            'supplier_currency_transfer_rate',
            'client_currency_id',
            'client_currency_transfer_rate',
            'extra_service_price',
            'extra_service_tree',
            'created',
            'modified',
            'purchase_price',
            'purchase_commission',
            'purchase_rate',
            'purchase_tax_id',
            'purchase_tax_amount',
            'purchase_currency_id',
            'purchase_tax_rate',
            'sale_price',
            'sale_commission',
            'sale_rate',
            'sale_tax_id',
            'sale_tax_amount',
            'sale_currency_id',
            'sale_tax_rate',
            'type',
            'model_type',
            'model_id',
            'profile_id',
        ];
    }

    private function relations(): array
    {
        return [
            'dailyProgram',
            'nationality.currentTranslation',
            'executive',
            'createdBy',
            'updatedBy',
            'salseUser',
            'currency.currentTranslation',
            'service.currentTranslation',
            'supplier.currentTranslation',
            'category.currentTranslation',
            'profile',
            'guide',
            'guideLanguage.currentTranslation',
            'taxes',
            'sellingTaxes',
            // Added for daily_transportation
            'company.currentTranslation',
            'city.currentTranslation',
            'cityTo.currentTranslation',
            'operationData.currentTranslation',
            // Added for accommodation search
            'reservation.accommodation.currentTranslation',
            'reservation.rooms',
            // Added for flights
            'segments.originAirport',
            'segments.destinationAirport',
            'segments.flightCompany',
            'airlineTickets.fromAirport',
            'airlineTickets.toAirport',
            'airlineTickets.fromAirline',
            'airlineTickets.toAirline',
        ];
    }

    public function getAllChanges($invoiceService)
    {
        return $invoiceService->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                // Collect related IDs from old/new values
                $companyIds = array_values(array_unique(array_filter([
                    $oldValues['company_id'] ?? null,
                    $newValues['company_id'] ?? null,
                ])));
                $supplierIds = array_values(array_unique(array_filter([
                    $oldValues['supplier_id'] ?? null,
                    $newValues['supplier_id'] ?? null,
                ])));
                $serviceIds = array_values(array_unique(array_filter([
                    $oldValues['service_id'] ?? null,
                    $newValues['service_id'] ?? null,
                ])));
                $currencyIds = array_values(array_unique(array_filter([
                    $oldValues['currency_id'] ?? null,
                    $newValues['currency_id'] ?? null,
                    $oldValues['purchase_currency_id'] ?? null,
                    $newValues['purchase_currency_id'] ?? null,
                    $oldValues['sale_currency_id'] ?? null,
                    $newValues['sale_currency_id'] ?? null,
                ])));
                $userIds = array_values(array_unique(array_filter([
                    $oldValues['executive_id'] ?? null,
                    $newValues['executive_id'] ?? null,
                    $oldValues['guide_id'] ?? null,
                    $newValues['guide_id'] ?? null,
                    $audit->user_id ?? null,
                ])));
                $nationalityIds = array_values(array_unique(array_filter([
                    $oldValues['nationality_id'] ?? null,
                    $newValues['nationality_id'] ?? null,
                ])));
                $guideLanguageIds = array_values(array_unique(array_filter([
                    $oldValues['guide_language_id'] ?? null,
                    $newValues['guide_language_id'] ?? null,
                ])));
                $cityIds = array_values(array_unique(array_filter([
                    $oldValues['city_id'] ?? null,
                    $newValues['city_id'] ?? null,
                ])));
                $operationDataIds = array_values(array_unique(array_filter([
                    $oldValues['operation_data_id'] ?? null,
                    $newValues['operation_data_id'] ?? null,
                ])));

                // Fetch names/maps
                $companies = ! empty($companyIds) ? Companies::whereIn('id', $companyIds)->pluck('name_company', 'id')->toArray() : [];
                $suppliers = ! empty($supplierIds) ? Suppliers::whereIn('id', $supplierIds)->pluck('name', 'id')->toArray() : [];
                $services = ! empty($serviceIds) ? GeneralService::whereIn('id', $serviceIds)->pluck('title', 'id')->toArray() : [];
                $currencies = ! empty($currencyIds) ? Currency::whereIn('id', $currencyIds)->pluck('code', 'id')->toArray() : [];
                $users = ! empty($userIds) ? User::whereIn('id', $userIds)->pluck('name', 'id')->toArray() : [];
                $nationalities = ! empty($nationalityIds) ? Nationality::whereIn('id', $nationalityIds)->pluck('name', 'id')->toArray() : [];
                $guideLanguages = ! empty($guideLanguageIds) ? GuideLanguage::whereIn('id', $guideLanguageIds)->pluck('title', 'id')->toArray() : [];
                $cities = ! empty($cityIds) ? City::whereIn('id', $cityIds)->pluck('name', 'id')->toArray() : [];
                $operations = ! empty($operationDataIds) ? OperationData::whereIn('id', $operationDataIds)->pluck('title', 'id')->toArray() : [];

                // Enrich readable names
                foreach (['company_id' => $companies, 'supplier_id' => $suppliers, 'service_id' => $services, 'currency_id' => $currencies, 'purchase_currency_id' => $currencies, 'sale_currency_id' => $currencies, 'executive_id' => $users, 'guide_id' => $users, 'nationality_id' => $nationalities, 'guide_language_id' => $guideLanguages, 'city_id' => $cities, 'operation_data_id' => $operations] as $key => $map) {
                    if (isset($oldValues[$key]) && isset($map[$oldValues[$key]])) {
                        $oldValues[$key.'_name'] = $map[$oldValues[$key]];
                    }
                    if (isset($newValues[$key]) && isset($map[$newValues[$key]])) {
                        $newValues[$key.'_name'] = $map[$newValues[$key]];
                    }
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }

    public function summeryinCurrency($profile, $currency)
    {
        /*
           return InvoiceServices::select(
               DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id) as final_currency_id'),
               DB::raw('MIN(currencies.name) as currency_name'),
               DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
               DB::raw('SUM(invoice_services.	grand_total) as 	grand_total'))
               ->join('currencies', 'currencies.id', '=', DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->where('invoice_services.profile_id', $profile)
               ->whereIn(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'), $currency)
               ->groupBy(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->groupBy('invoice_services.currency_id', 'currencies.name')
               ->get();
        */
        return InvoiceServices::select(
            'invoice_services.currency_id as final_currency_id',
            DB::raw('MIN(currencies.name) as currency_name'),
            DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
            DB::raw('SUM(invoice_services.grand_total) as grand_total')
        )
            ->join('currencies', 'currencies.id', '=', 'invoice_services.currency_id')
            ->where('invoice_services.profile_id', $profile)
            ->whereIn('invoice_services.currency_id', $currency)
            ->groupBy('invoice_services.currency_id')
            ->get();

    }

    public function summeryinCurrencyNotIn($profile, $currency)
    {

        /*
           return InvoiceServices::select(
               DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id) as final_currency_id'),
               DB::raw('MIN(currencies.name) as currency_name'),
               DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
               DB::raw('SUM(invoice_services.	grand_total) as 	grand_total'))
               ->join('currencies', 'currencies.id', '=', DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->where('invoice_services.profile_id', $profile)
               ->whereNotIn(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'), $currency)
               ->groupBy(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->groupBy('invoice_services.currency_id', 'currencies.name')
               ->get();
        */
        return InvoiceServices::select(
            'invoice_services.currency_id as final_currency_id',
            DB::raw('MIN(currencies.name) as currency_name'),
            DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
            DB::raw('SUM(invoice_services.grand_total) as grand_total')
        )
            ->join('currencies', 'currencies.id', '=', 'invoice_services.currency_id')
            ->where('invoice_services.profile_id', $profile)
            ->whereNotIn('invoice_services.currency_id', $currency)
            ->groupBy('invoice_services.currency_id')
            ->get();

    }

    public function CheckInvoiceServices($services, $profile)
    {
        return InvoiceServices::whereIn('travel_tourism_type', $services)->where('profile_id', $profile)->exists();
    }

    public function getServicesForProfile($profile)
    {
        $services = ['tour_guide', 'tour_reps', 'daily_transportation', 'dining_entertainment', 'daily_program_id'];

        return InvoiceServices::with([
            'dailyProgram',
            'service.currentTranslation',
            'guideLanguage.currentTranslation',
            'nationality.currentTranslation',
            'city.currentTranslation',
            'operationData.currentTranslation',
        ])
            ->whereIn('travel_tourism_type', $services)
            ->where('profile_id', $profile)
            ->orderBy('daily_program_id', 'asc')
            ->get()
            ->groupBy('daily_program_id')
            ->map(function ($group) {
                $dailyProgram = $group->first()->dailyProgram;

                return [
                    'day_number' => $dailyProgram->day_number ?? null,
                    'day_date' => $dailyProgram->day_date ?? null,
                    'services' => $group->map(function ($item) {
                        return [
                            'title' => $item->service->currentTranslation->title ?? $item->service->title,
                            'type' => $item->service->type ?? null,
                            'notes' => $item->notes ?? null,
                            'people_count' => $item->people_count ?? null,
                            'guide_language_name' => $item->guideLanguage->currentTranslation->name
                                ?? $item->guideLanguage->name
                                    ?? null,
                            'nationality_id' => $item->nationality_id ?? null,
                            'nationality_name' => $item->nationality->currentTranslation->name
                                ?? $item->nationality->name
                                    ?? null,
                            'adults_count' => $item->adults_count ?? null,
                            'children_count' => $item->children_count ?? null,
                            'city_id' => $item->city_id ?? null,
                            'city_name' => $item->city->currentTranslation->name
                                ?? $item->city->name
                                    ?? null,
                            'operation_data_id' => $item->operation_data_id ?? null,
                            'operation_data_name' => $item->operationData->currentTranslation->name
                                ?? $item->operationData->name
                                    ?? null,
                        ];
                    })->values(),
                ];
            })->values();
    }

    public function getOtherSerForProfile($services, $profile)
    {
        //   return InvoiceServices::with('service','city','cityTo')->where('travel_tourism_type', $services)->where('profile_id',$profile)->get();
        /*
        return InvoiceServices::with([
            'service:id,title',
            'city:id,name',
            'cityTo:id,name',
        ])
            ->where('travel_tourism_type', $services)
            ->where('profile_id', $profile)
            ->get(['id', 'service_id', 'city_id', 'city_to_id', 'people_count'])
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'service_title' => $item->service->title ?? null,
                    'city_name' => $item->city->name ?? null,
                    'city_to_name' => $item->cityTo->name ?? null,
                    'people_count' => $item->people_count,
                ];
            });
        */
        return InvoiceServices::with(['cityTo:id,name',
            'nationality.currentTranslation',
            'operationData.currentTranslation', ])
            ->where('travel_tourism_type', $services)
            ->where('profile_id', $profile)
            ->get(['id',
                'service_id',
                'city_id',
                'city_to_id',
                'people_count',
                'nationality_id',
                'operation_data_id',
                'execution_date',
                'notes', ])
            ->map(function ($item) {
                return [
                    // المطلوبين
                    'city_to_name' => $item->cityTo->name ?? null,
                    'people_count' => $item->people_count,

                    'nationality_id' => $item->nationality_id ?? null,
                    'nationality_name' => $item->nationality->currentTranslation->name
                        ?? $item->nationality->name
                            ?? null,

                    'execution_date' => $item->execution_date,

                    'operation_data_id' => $item->operation_data_id ?? null,
                    'operation_data_name' => $item->operationData->currentTranslation->name
                        ?? $item->operationData->name
                            ?? null,

                    'city_to_id' => $item->city_to_id,
                    'notes' => $item->notes,
                ];
            });
    }

    public function search()
    {
        // Define all filter variables first
        $type = request('type', null);
        // Handle type as string, array, or comma-separated string
        if ($type) {
            if (is_string($type)) {
                $type = strpos($type, ',') !== false ? explode(',', $type) : [$type];
            } elseif (! is_array($type)) {
                $type = [$type];
            }
        }
        $supplier_name = request('supplier_name', null);
        $profile_nam = request('profile_num', null);
        $currency = request('currency_id', null);
        $accommodation_reservation_status = request('status', null);
        $arrivalDateFrom = request('arrival_date_from', null);
        $arrivalDateTo = request('arrival_date_to', null);
        $checkin = request('checkin', null); // الهوتيل
        $checkout = request('checkout', null);
        $execution_date = request('execution_date', null); // خدمات الاخري
        $daily_date = request('daily_date', null); // services
        $supplier_id = request('supplier_id', null);
        $serviceName = request('service_name', null);
        $cityFrom = request('city_from', null);
        $cityTo = request('city_to', null);
        $guideLanguage = request('guide_language', null);
        $hotel_name = request('accommodation_name', null);
        $limit = request('limit', 20);
        $is_by_handling = request('is_by_handling', null);
        $next_date_from = request('next_date_from', null);
        $next_date_to = request('next_date_to', null);
        $executive_id = request('executive_id', null);
        $invoice_services_id = request('invoice_service_id', null);
        $room_id = request('room_id', null);
        $created_by = request('created_by', null);
        $updated_by = request('updated_by', null);
        $salse_id = request('salse_id', null);
        $servics_ids = Auth()->user()->services()->select('services.id')->pluck('id')->toArray();
        $accommodation_ids = Auth()->user()->accommodations()->select('ac_accommodations.id')->pluck('id')->toArray();

        $sendSerivcseId = request('services_id', null);
        $sendAccommodationsId = request('accommodation_id', null);
        $query = InvoiceServices::with($this->relations())->select('invoice_services.*');

        // 1. Accommodation type filter (hotel/cruise) through reservation relationship
        $query->when($type, function ($q) use ($type) {
            // Convert to array if it's a string
            $types = is_array($type) ? $type : [$type];

            $reservationTypes = array_intersect($types, ['hotel', 'cruise']);
            $otherTypes = array_diff($types, ['hotel', 'cruise']);

            // If we have both accommodation types and other types, use OR condition
            if (! empty($reservationTypes) && ! empty($otherTypes)) {
                return $q->where(function ($subQ) use ($reservationTypes, $otherTypes) {
                    // Search in reservations for hotel/cruise
                    $subQ->whereHas('reservation', function ($resQ) use ($reservationTypes) {
                        $resQ->whereHas('accommodation', function ($accQ) use ($reservationTypes) {
                            $accQ->whereIn('type', $reservationTypes);
                        });
                    })
                        // OR search in travel_tourism_type for other types
                        ->orWhereIn('travel_tourism_type', $otherTypes);
                });
            } // If only accommodation types
            elseif (! empty($reservationTypes)) {
                return $q->whereHas('reservation', function ($subQ) use ($reservationTypes) {
                    return $subQ->whereHas('accommodation', function ($subQ) use ($reservationTypes) {
                        $subQ->whereIn('type', $reservationTypes);
                    });
                });
            } // If only other types
            else {
                return $q->whereIn('travel_tourism_type', $otherTypes);
            }
        });

        $query->when($hotel_name, function ($q) use ($hotel_name) {
            return $q->whereHas('reservation', function ($subQ) use ($hotel_name) {
                $subQ->whereHas('accommodation', function ($subQ) use ($hotel_name) {
                    $subQ->where('name', 'like', "%$hotel_name%");
                });
            });
        });
        /*
        $query->when($accommodation_reservation_status, function ($q) use ($accommodation_reservation_status) {
            return $q->whereHas('reservation', function ($subQ) use ($accommodation_reservation_status) {
                $subQ->where('status', $accommodation_reservation_status);
            });
        });
            */
        $query->when($accommodation_reservation_status, function ($q) use ($accommodation_reservation_status) {
            return $q->where('invoice_services.status', $accommodation_reservation_status);
        });

        // 2. Profile ID filter
        $query->when($currency, function ($q) use ($currency) {
            return $q->where('currency_id', $currency);
        });

        $query->when($supplier_id, function ($q) use ($supplier_id) {
            return $q->where('supplier_id', $supplier_id);
        });

        $query->when($is_by_handling, function ($q) use ($is_by_handling) {
            return $q->where('is_by_handling', $is_by_handling);
        });

        // Handle next_date_from and next_date_to separately
        $query->when($next_date_from && $next_date_to, function ($q) use ($next_date_from, $next_date_to) {
            // Both dates provided: use whereBetween
            return $q->whereBetween('next_pay_date', [$next_date_from, $next_date_to]);
        })->when($next_date_from && ! $next_date_to, function ($q) use ($next_date_from) {
            // Only from date: next_pay_date >= next_date_from
            return $q->where('next_pay_date', '>=', $next_date_from);
        })->when($next_date_to && ! $next_date_from, function ($q) use ($next_date_to) {
            // Only to date: next_pay_date <= next_date_to
            return $q->where('next_pay_date', '<=', $next_date_to);
        });

        $query->when($supplier_name, function ($q) use ($supplier_name) {
            return $q->whereHas('supplier', function ($subQ) use ($supplier_name) {
                $subQ->where('supplier_name', 'like', "%$supplier_name%");
            });
        });

        // 3. Profile arrival and departure dates
        // Filter profiles that overlap with the date range
        // If both dates are provided, find profiles that intersect with the range
        // If only from date, find profiles that depart after or on that date
        // If only to date, find profiles that arrive before or on that date
        $query->when($arrivalDateFrom || $arrivalDateTo, function ($q) use ($arrivalDateFrom, $arrivalDateTo) {
            return $q->whereHas('profile', function ($subQ) use ($arrivalDateFrom, $arrivalDateTo) {
                if ($arrivalDateFrom && $arrivalDateTo) {
                    // Date range: find profiles that overlap with the range
                    // Profile arrives before or on the end date AND departs after or on the start date
                    $subQ->where('arrival_date', '<=', $arrivalDateTo)
                        ->where('departure_date', '>=', $arrivalDateFrom);
                } elseif ($arrivalDateFrom) {
                    // Only from date: find profiles that depart after or on this date
                    $subQ->where('departure_date', '>=', $arrivalDateFrom);
                } elseif ($arrivalDateTo) {
                    // Only to date: find profiles that arrive before or on this date
                    $subQ->where('arrival_date', '<=', $arrivalDateTo);
                }
            });
        });

        $query->when($profile_nam, function ($q) use ($profile_nam) {
            return $q->whereHas('profile', function ($subQ) use ($profile_nam) {
                $subQ->where('profile_number', 'like', "%$profile_nam%");
            });
        });

        // 4. Service name filter
        $query->when($serviceName, function ($q) use ($serviceName) {
            return $q->whereHas('service', function ($subQ) use ($serviceName) {
                $subQ->where('title', 'like', '%'.$serviceName.'%');
            });
        });

        // 5. City filters (from and to)
        $query->when($cityFrom, function ($q) use ($cityFrom) {
            return $q->where('city_id', $cityFrom);
        });

        $query->when($cityTo, function ($q) use ($cityTo) {
            return $q->where('city_to_id', $cityTo);
        });
        $query->when($execution_date, function ($q) use ($execution_date) {
            return $q->where('execution_date', $execution_date);
        });
        $query->when($daily_date, function ($q) use ($daily_date) {
            return $q->whereHas('dailyProgram', function ($subQ) use ($daily_date) {
                $subQ->where('day_date', $daily_date);
            });
        });

        // Filter by check_in - use model relation directly with >= comparison
        $query->when($checkin, function ($q) use ($checkin) {
            return $q->whereHas('model', function ($subQ) use ($checkin) {
                $subQ->where('check_in', '>=', $checkin);
            });
        });

        // Filter by check_out - use model relation directly with <= comparison
        $query->when($checkout, function ($q) use ($checkout) {
            return $q->whereHas('model', function ($subQ) use ($checkout) {
                $subQ->where('check_out', '<=', $checkout);
            });
        });

        // 6. Guide language filter
        $query->when($guideLanguage, function ($q) use ($guideLanguage) {
            return $q->where('guide_language_id', $guideLanguage);
        });
        // Additional filters - User's services and accommodations (OR condition)
        if (! empty($servics_ids) || ! empty($accommodation_ids)) {
            $query->where(function ($q) use ($servics_ids, $accommodation_ids) {
                if (! empty($servics_ids)) {
                    $q->whereIn('service_id', $servics_ids);
                }
                if (! empty($accommodation_ids)) {
                    $q->orWhereHas('reservation', function ($subQ) use ($accommodation_ids) {
                        $subQ->whereIn('accommodation_id', $accommodation_ids);
                    });
                }
            });
        }

        $userGroupIds = auth()->user()->groups()->select('groups.id')->pluck('id')->toArray();
        $hasHandling = in_array(GroupEnum::HANDLING->value, $userGroupIds);
        $hasHandlingManager = in_array(GroupEnum::HANDLING_MANAGER->value, $userGroupIds);

        if ($hasHandling && ! $hasHandlingManager) {
            $query->where('executive_id', auth()->id());
        }

        $query->when($executive_id, function ($q) use ($executive_id) {
            return $q->where('executive_id', $executive_id);
        });
        $query->when($sendSerivcseId, function ($q) use ($sendSerivcseId) {
            return $q->where('service_id', $sendSerivcseId);
        });
        $query->when($sendAccommodationsId, function ($q) use ($sendAccommodationsId) {
            return $q->whereHas('reservation', function ($subQ) use ($sendAccommodationsId) {
                $subQ->whereIn('accommodation_id', $sendAccommodationsId);
            });
        });

        // Filter by invoice_services_id (search by id)
        $query->when($invoice_services_id, function ($q) use ($invoice_services_id) {
            return $q->where('id', $invoice_services_id);
        });

        // Filter by room_id (search by model_id)
        $query->when($room_id, function ($q) use ($room_id) {
            return $q->where('model_id', $room_id);
        });

        // Filter by created_by
        $query->when($created_by, function ($q) use ($created_by) {
            return $q->where('created_by', $created_by);
        });

        // Filter by updated_by
        $query->when($updated_by, function ($q) use ($updated_by) {
            return $q->where('updated_by', $updated_by);
        });

        // Filter by salse_id
        $query->when($salse_id, function ($q) use ($salse_id) {
            return $q->where('salse_id', $salse_id);
        });

        // Sorting
        $sortBy = request('sort_by');
        $sortOrder = $this->normalizeSortOrder(request('sort_order', 'desc'));

        if ($sortBy) {
            $hasJoins = $this->applySorting($query, $sortBy, $sortOrder);
        } else {
            // Default sorting
            $query->orderByDesc('invoice_services.id');
        }

        // Only use select() if we have joins to avoid column conflicts
        // Using selectRaw() ensures proper column selection when joins are present
        // Eager loading (with()) works independently and won't be affected

        // Pagination
        $results = $query->paginate($limit);

        // Transform results to include structured data
        $results->getCollection()->transform(function ($item) {
            $lastAudit = $item->audits()->with('user')->latest()->first();

            return [
                'id' => $item->id,
                'type' => $item->travel_tourism_type == 'accommodation' ? $this->getHotelOrCruiseName($item) : $item->travel_tourism_type,
                'title' => $item->travel_tourism_type == 'accommodation' ? $this->getAccommodationName($item) : ($item->service && $item->service->currentTranslation ? $item->service->currentTranslation->title : null),
                'profile_id' => $item->profile_id,
                'profile_number' => $item->profile->profile_number ?? null,
                'arrival_date' => $item->profile && $item->profile->arrival_date ? date('Y-m-d', strtotime($item->profile->arrival_date)) : null,
                'departure_date' => $item->profile && $item->profile->departure_date ? date('Y-m-d', strtotime($item->profile->departure_date)) : null,
                'city_from' => $item->city->currentTranslation->name ?? null,
                'city_to' => $item->cityTo->currentTranslation->name ?? null,
                'guide_language' => $item->guideLanguage->currentTranslation->name ?? null,
                'execution_date' => $item->daily_program_id === null
                    ? ($item->execution_date ? (is_string($item->execution_date) ? date('Y-m-d', strtotime($item->execution_date)) : $item->execution_date->format('Y-m-d')) : null)
                    : ($item->dailyProgram && $item->dailyProgram->day_date ? (is_string($item->dailyProgram->day_date) ? date('Y-m-d', strtotime($item->dailyProgram->day_date)) : $item->dailyProgram->day_date->format('Y-m-d')) : null),
                'grand_total' => $item->travel_tourism_type == 'accommodation' ? $item->purchase_price : $item->grand_total,
                'total_paid' => 0,
                'remainder' => $item->remaining_to_supplier,
                'paid_to_supplier' => $item->paid_to_supplier,
                'is_paid' => $item->is_paid ?? 0,
                'notes' => $item->notes,
                'supplier_name' => $item->supplier->currentTranslation->supplier_name ?? null,
                'supplier_id' => $item->supplier_id,
                'reset_num' => $item->reservation->reservation_num ?? null,
                'accommodation_confirm_number' => $item->reservation->confirmation_num ?? null,
                'daily_program_id' => $item->daily_program_id,
                'currency_id' => $item->currency_id,
                'currency_name' => $item->currency->currentTranslation->name ?? null,
                'executive_id' => $item->executive_id,
                'executive_name' => $item->executive ? $item->executive->name : null,
                'reservation_id' => $item->reservation_id ?? null,
                'room_id' => $item->model_id ?? null,
                'next_pay_date' => $item->next_pay_date ?? null,
                'created_by' => $item->created_by,
                'created_by_name' => $item->createdBy ? $item->createdBy->name : null,
                'updated_by' => $item->updated_by,
                'updated_by_name' => $item->updatedBy ? $item->updatedBy->name : ($lastAudit?->user?->name ?? null),
                'salse_id' => $item->salse_id,
                'salse_name' => $item->salseUser ? $item->salseUser->name : null,
                'updated_at' => $item->modified ? date('Y-m-d H:i:s', strtotime($item->modified)) : null,
                'status' => $item->status,
                'check_in' => $item->travel_tourism_type == 'accommodation' && $item->reservation && $item->reservation->relationLoaded('rooms') && $item->reservation->rooms->isNotEmpty() ? ($item->reservation->rooms->min('check_in') ? date('Y-m-d', strtotime($item->reservation->rooms->min('check_in'))) : null) : null,
                'check_out' => $item->travel_tourism_type == 'accommodation' && $item->reservation && $item->reservation->relationLoaded('rooms') && $item->reservation->rooms->isNotEmpty() ? ($item->reservation->rooms->max('check_out') ? date('Y-m-d', strtotime($item->reservation->rooms->max('check_out'))) : null) : null,
            ];
        });

        // Return pagination with transformed data
        return [
            'data' => $results->items(),
            'pagination' => [
                'current_page' => $results->currentPage(),
                'last_page' => $results->lastPage(),
                'per_page' => $results->perPage(),
                'total' => $results->total(),
                'from' => $results->firstItem(),
                'to' => $results->lastItem(),
                'has_more_pages' => $results->hasMorePages(),
            ],
        ];
    }

    /**
     * Normalize sort order to 'asc' or 'desc'
     */
    private function normalizeSortOrder(?string $order): string
    {
        return in_array(strtolower($order ?? ''), ['asc', 'desc']) ? strtolower($order) : 'desc';
    }

    /**
     * Apply sorting to query based on sort_by field
     * Returns true if joins were added (needs select()), false otherwise
     */
    private function applySorting($query, string $sortBy, string $sortOrder): bool
    {
        $sortMapping = $this->getSortMapping();

        if (! isset($sortMapping[$sortBy])) {
            // Invalid sort_by, use default
            $query->orderByDesc('invoice_services.modified');

            return false;
        }

        $sortConfig = $sortMapping[$sortBy];

        // Handle aggregate fields (check_in, check_out) - uses joins
        if (isset($sortConfig['aggregate'])) {
            $this->applyAggregateSorting($query, $sortBy, $sortConfig['aggregate'], $sortOrder);

            return true; // Has joins
        }

        // Handle relationship fields - uses joins
        if (isset($sortConfig['relationship'])) {
            $hasJoins = $this->applyRelationshipSorting($query, $sortConfig['relationship'], $sortConfig['column'], $sortOrder);

            return $hasJoins;
        }

        // Handle direct column fields - no joins needed
        if (isset($sortConfig['column'])) {
            $this->applyDirectSorting($query, $sortConfig['column'], $sortOrder);

            return false; // No joins
        }

        // Fallback to default
        $query->orderByDesc('invoice_services.modified');

        return false;
    }

    /**
     * Apply aggregate sorting (MIN/MAX from related tables)
     */
    private function applyAggregateSorting($query, string $sortBy, string $aggregate, string $sortOrder): void
    {
        $aggregateMap = [
            'check_in' => ['table' => 'pr_accommodation_reservation_rooms', 'column' => 'check_in', 'function' => 'MIN', 'alias' => 'rooms_min', 'join_column' => 'min_check_in', 'foreign_key' => 'reservation_id'],
            'check_out' => ['table' => 'pr_accommodation_reservation_rooms', 'column' => 'check_out', 'function' => 'MAX', 'alias' => 'rooms_max', 'join_column' => 'max_check_out', 'foreign_key' => 'reservation_id'],
        ];

        if (! isset($aggregateMap[$sortBy])) {
            return;
        }

        $config = $aggregateMap[$sortBy];
        $subquery = DB::table("{$config['table']} as rooms")
            ->select("rooms.{$config['foreign_key']}")
            ->selectRaw("{$config['function']}(rooms.{$config['column']}) as {$config['join_column']}")
            ->groupBy("rooms.{$config['foreign_key']}");

        $query->leftJoinSub($subquery, $config['alias'], function ($join) use ($config) {
            $join->on('invoice_services.reservation_id', '=', "{$config['alias']}.{$config['foreign_key']}");
        });

        $this->applyOrderBy($query, "{$config['alias']}.{$config['join_column']}", $sortOrder);
    }

    /**
     * Apply relationship sorting (join with related tables)
     * Returns true if joins were added, false otherwise
     */
    private function applyRelationshipSorting($query, string $relationship, string $column, string $sortOrder): bool
    {
        $relationshipParts = explode('.', $relationship);

        // Handle nested relationships (e.g., 'reservation.rooms') - use subquery
        if (count($relationshipParts) === 2) {
            return $this->applyNestedRelationshipSorting($query, $relationshipParts, $column, $sortOrder);
        }

        // Handle single-level relationships
        $relation = $relationshipParts[0];
        $table = $this->getRelationTable($relation);

        if (! $table) {
            return false;
        }

        // Handle polymorphic model relation (only for rooms)
        if ($relation === 'model') {
            $query->leftJoin($table, 'invoice_services.model_id', '=', "{$table}.id");
        } else {
            $foreignKey = $this->getRelationForeignKey($relation);
            $query->leftJoin($table, "invoice_services.{$foreignKey}", '=', "{$table}.id");
        }

        // Handle translation tables for name fields
        $sortColumn = $this->getSortColumnForRelationship($query, $relation, $table, $column);

        $this->applyOrderBy($query, $sortColumn, $sortOrder);

        return true; // Has joins
    }

    /**
     * Apply nested relationship sorting using subquery
     * Returns true if joins were added, false otherwise
     */
    private function applyNestedRelationshipSorting($query, array $relationshipParts, string $column, string $sortOrder): bool
    {
        [$parentRelation, $childRelation] = $relationshipParts;

        // Currently only support reservation.rooms for check_in/check_out (handled in aggregate)
        // For other nested relationships, fallback to default sorting
        $query->orderByDesc('invoice_services.modified');

        return false; // No joins added
    }

    /**
     * Get sort column for relationship (handles translation tables)
     */
    private function getSortColumnForRelationship($query, string $relation, string $table, string $column): string
    {
        // Check if this is a translated field (supplier name, service title, guide language name, city name)
        // Note: currency uses direct name column, not translation table
        $isTranslatedField = ($relation === 'supplier' && strpos($column, 'name') !== false)
            || ($relation === 'service' && $column === 'title')
            || ($relation === 'guideLanguage' && $column === 'name')
            || ($relation === 'city' && $column === 'name')
            || ($relation === 'cityTo' && $column === 'name');

        if (! $isTranslatedField) {
            return "{$table}.{$column}";
        }

        $langId = app('lang_id');
        $translationConfig = [
            'supplier' => ['table' => 'su_supplier_translation', 'foreign_key' => 'supplier_id', 'column' => 'supplier_name'],
            'service' => ['table' => 'services_translation', 'foreign_key' => 'services_id', 'column' => 'title'],
            'guideLanguage' => ['table' => 'guide_language_translation', 'foreign_key' => 'guide_language_id', 'column' => 'name'],
            'city' => ['table' => 'city_translation', 'foreign_key' => 'city_id', 'column' => 'name'],
            'cityTo' => ['table' => 'city_translation', 'foreign_key' => 'city_id', 'column' => 'name'],
        ];

        if (! isset($translationConfig[$relation])) {
            return "{$table}.{$column}";
        }

        $transConfig = $translationConfig[$relation];
        $translationTable = $transConfig['table'];

        $query->leftJoin($translationTable, function ($join) use ($table, $translationTable, $transConfig, $langId) {
            $join->on("{$table}.id", '=', "{$translationTable}.{$transConfig['foreign_key']}")
                ->where("{$translationTable}.lang_id", '=', $langId);
        });

        return "{$translationTable}.{$transConfig['column']}";
    }

    /**
     * Apply direct column sorting
     */
    private function applyDirectSorting($query, string $column, string $sortOrder): void
    {
        $this->applyOrderBy($query, $column, $sortOrder);
    }

    /**
     * Apply orderBy based on sort order
     */
    private function applyOrderBy($query, string $column, string $sortOrder): void
    {
        $sortOrder === 'asc' ? $query->orderBy($column) : $query->orderByDesc($column);
    }

    /**
     * Get relation table name for sorting
     */
    private function getRelationTable(string $relation): ?string
    {
        $tables = [
            'profile' => 'pr_profile',
            'supplier' => 'su_supplier',
            'currency' => 'currencies',
            'executive' => 'users',
            'createdBy' => 'users',
            'updatedBy' => 'users',
            'salseUser' => 'users',
            'service' => 'services',
            'model' => 'pr_accommodation_reservation_rooms',
            'guideLanguage' => 'guide_language',
            'city' => 'cities',
            'cityTo' => 'cities',
            'reservation' => 'pr_accommodation_reservation',
        ];

        return $tables[$relation] ?? null;
    }

    /**
     * Get foreign key column name for relation
     */
    private function getRelationForeignKey(string $relation): string
    {
        $foreignKeys = [
            'profile' => 'profile_id',
            'supplier' => 'supplier_id',
            'currency' => 'currency_id',
            'executive' => 'executive_id',
            'createdBy' => 'created_by',
            'updatedBy' => 'updated_by',
            'salseUser' => 'salse_id',
            'service' => 'service_id',
            'guideLanguage' => 'guide_language_id',
            'city' => 'city_id',
            'cityTo' => 'city_to_id',
            'reservation' => 'reservation_id',
        ];

        return $foreignKeys[$relation] ?? $relation.'_id';
    }

    /**
     * Get sort mapping for return properties to database columns
     */
    private function getSortMapping(): array
    {
        return [
            // Direct columns
            'id' => ['column' => 'invoice_services.id'],
            'profile_id' => ['column' => 'invoice_services.profile_id'],
            'grand_total' => ['column' => 'invoice_services.grand_total'],
            'remainder' => ['column' => 'invoice_services.remaining_to_supplier'],
            'paid_to_supplier' => ['column' => 'invoice_services.paid_to_supplier'],
            'is_paid' => ['column' => 'invoice_services.is_paid'],
            'supplier_id' => ['column' => 'invoice_services.supplier_id'],
            'daily_program_id' => ['column' => 'invoice_services.daily_program_id'],
            'currency_id' => ['column' => 'invoice_services.currency_id'],
            'executive_id' => ['column' => 'invoice_services.executive_id'],
            'reservation_id' => ['column' => 'invoice_services.reservation_id'],
            'room_id' => ['column' => 'invoice_services.model_id'],
            'next_pay_date' => ['column' => 'invoice_services.next_pay_date'],
            'created_by' => ['column' => 'invoice_services.created_by'],
            'updated_by' => ['column' => 'invoice_services.updated_by'],
            'salse_id' => ['column' => 'invoice_services.salse_id'],
            'status' => ['column' => 'invoice_services.status'],
            'updated_at' => ['column' => 'invoice_services.modified'],
            'execution_date' => ['column' => 'invoice_services.execution_date'],

            // Relationships - need joins
            'profile_number' => [
                'relationship' => 'profile',
                'column' => 'profile_number',
            ],
            'arrival_date' => [
                'relationship' => 'profile',
                'column' => 'arrival_date',
            ],
            'departure_date' => [
                'relationship' => 'profile',
                'column' => 'departure_date',
            ],
            'supplier_name' => [
                'relationship' => 'supplier',
                'column' => 'supplier_name',
            ],
            'currency_name' => [
                'relationship' => 'currency',
                'column' => 'name',
            ],
            'executive_name' => [
                'relationship' => 'executive',
                'column' => 'name',
            ],
            'created_by_name' => [
                'relationship' => 'createdBy',
                'column' => 'name',
            ],
            'updated_by_name' => [
                'relationship' => 'updatedBy',
                'column' => 'name',
            ],
            'salse_name' => [
                'relationship' => 'salseUser',
                'column' => 'name',
            ],
            'title' => [
                'relationship' => 'service',
                'column' => 'title',
            ],
            'guide_language' => [
                'relationship' => 'guideLanguage',
                'column' => 'name',
            ],
            'city_from' => [
                'relationship' => 'city',
                'column' => 'name',
            ],
            'city_to' => [
                'relationship' => 'cityTo',
                'column' => 'name',
            ],
            'accommodation_confirm_number' => [
                'relationship' => 'reservation',
                'column' => 'confirmation_num',
            ],
            'reset_num' => [
                'relationship' => 'reservation',
                'column' => 'reservation_num',
            ],

            // Special cases - computed fields (sort by related column)
            'total_paid' => ['column' => 'invoice_services.paid_to_supplier'], // total_paid is always 0, so sort by paid_to_supplier
            'check_in' => [
                'relationship' => 'model',
                'column' => 'check_in',
            ],
            'check_out' => [
                'relationship' => 'model',
                'column' => 'check_out',
            ],
        ];
    }

    /**
     * Get room name from accommodation reservation
     */
    private function getRoomNum($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $room = $item->model;
            if ($room) {
                return $room->room_number ?? null;
            }
        }

        return null;
    }

    private function getRoomType($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $room = $item->model;
            if ($room && $room->roomType) {
                return $room->roomType->currentTranslation->name ?? $room->roomType->name ?? null;
            }
        }

        return null;
    }

    private function getRoomView($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $roomView = $item->model->roomView;
            if ($roomView) {
                return $roomView->currentTranslation->name ?? $roomView->name ?? null;
            }
        }

        return null;
    }

    private function getRoomMeal($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $mealPlan = $item->model->mealPlan;
            if ($mealPlan) {
                return $mealPlan->currentTranslation->name ?? $mealPlan->name ?? null;
            }
        }

        return null;
    }

    /**
     * Get accommodation name
     */
    private function getAccommodationName($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->reservation && $item->reservation->accommodation) {
            return $item->reservation->accommodation->currentTranslation->name ?? null;
        }

        return null;
    }

    /**
     * Get hotel or cruise name based on accommodation type
     */
    private function getHotelOrCruiseName($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->reservation) {
            $accommodation = $item->reservation->accommodation;
            if ($accommodation) {
                $type = $item->reservation->accommodation->type; // hotel or cruise

                return $type;
            }
        }

        return null;
    }

    public function getHandlingServices(array $filters = [])
    {
        // 1. Remove ->query() from here
        $query = InvoiceServices::with($this->relations())->where('is_by_handling', 1);
        if (isset($filters['profile_id'])) {
            $query->where('profile_id', $filters['profile_id']);
        }

        if (isset($filters['travel_tourism_type'])) {
            $query->where('travel_tourism_type', $filters['travel_tourism_type']);
        }

        if (isset($filters['service_id'])) {
            $query->where('service_id', $filters['service_id']);
        }

        if (isset($filters['invoice_id'])) {
            $query->where('invoice_id', $filters['invoice_id']);
        }

        // 2. This is the corrected 'executive_id' filter
        if (isset($filters['executive_id'])) {
            $query->where('executive_id', $filters['executive_id']);
        }

        // 3. The repeated invoice_id check was removed.

        // Add pagination if needed
        $limit = $filters['limit'] ?? 15;

        return $query->paginate($limit);
    }

    /**
     * Search flights without profile_id using flexible filters.
     */
    public function searchFlightsWithoutProfile(array $filters, int $limit = 20)
    {
        $query = InvoiceServices::query()
            ->with([
                'segments:id,invoice_services_id,origin_airport,destination_airport,date,time,type,flight_number,flight_company',
            ])
            ->where('travel_tourism_type', 'flights')
            ->whereNull('profile_id');

        $query->when($filters['pnr_number'] ?? null, function ($q, string $pnr) {
            return $q->where('pnr_number', 'like', '%'.$pnr.'%');
        });

        $query->when($filters['office_id'] ?? null, function ($q, $officeId) {
            return $q->where('office_id', $officeId);
        });

        $query->when($filters['service_id'] ?? null, function ($q, $serviceId) {
            return $q->where('service_id', $serviceId);
        });

        $query->when(($filters['origin_airport'] ?? null) || ($filters['origin_date'] ?? null), function ($q) use ($filters) {
            $q->whereHas('segments', function ($segmentQuery) use ($filters) {
                $segmentQuery->where('type', 'go')
                    ->when($filters['origin_airport'] ?? null, function ($segmentQuery, $originAirport) {
                        return $segmentQuery->where('origin_airport', $originAirport);
                    })
                    ->when($filters['origin_date'] ?? null, function ($segmentQuery, $originDate) {
                        return $segmentQuery->whereDate('date', $originDate);
                    });
            });
        });

        $query->when(($filters['destination_airport'] ?? null) || ($filters['destination_date'] ?? null), function ($q) use ($filters) {
            $q->whereHas('segments', function ($segmentQuery) use ($filters) {
                $segmentQuery->where('type', 'return')
                    ->when($filters['destination_airport'] ?? null, function ($segmentQuery, $destinationAirport) {
                        return $segmentQuery->where('destination_airport', $destinationAirport);
                    })
                    ->when($filters['destination_date'] ?? null, function ($segmentQuery, $destinationDate) {
                        return $segmentQuery->whereDate('date', $destinationDate);
                    });
            });
        });

        $query->when($filters['flight_company'] ?? null, function ($q, $flightCompany) {
            $q->whereHas('segments', function ($segmentQuery) use ($flightCompany) {
                return $segmentQuery->where('flight_company', $flightCompany);
            });
        });

        return $query
            ->orderByDesc('created')
            ->paginate($limit);
    }

    /**
     * Bulk-assign profile data to invoice services.
     */
    public function assignProfileToServices(array $ids, Profile $profile): array
    {
        $payload = [
            'profile_id' => $profile->id,
            'salse_id' => $profile->user_id,
            'is_by_handling' => $profile->has_invoice ? 1 : 0,
            'modified' => now(),
        ];

        DB::transaction(function () use ($ids, $payload) {
            InvoiceServices::whereIn('id', $ids)->update($payload);
        });

        return [
            'profile_id' => $profile->id,
            'updated_ids' => $ids,
            'updated_count' => count($ids),
        ];
    }

    public function GetSummary($profileId)
    {

        return InvoiceServices::select(
            'invoice_services.currency_id as final_currency_id',
            DB::raw('MIN(currencies.name) as currency_name'),
            DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
            DB::raw('SUM(invoice_services.grand_total) as grand_total'),
            DB::raw('COUNT(invoice_services.id) as services_count')
        )
            ->join('currencies', 'currencies.id', '=', 'invoice_services.currency_id')
            ->where('invoice_services.profile_id', $profileId)
            ->where('invoice_services.is_by_handling', 1)
            ->groupBy('invoice_services.currency_id')
            ->get();

    }

    /**
     * Create purchase invoice constraint (قيد استلام الخدمة من المورد)
     */
    public function createPurchaseInvoiceConstraint(InvoiceServices $invoiceService)
    {
        // Skip if no service_id or currency_id
        if (! $invoiceService->service_id || ! $invoiceService->currency_id) {
            return false;
        }

        // Get service cost account from services_currencies
        $serviceCurrency = ServicesCurrencies::where('service_id', $invoiceService->service_id)
            ->where('currency_id', $invoiceService->currency_id)
            ->first();

        if (! $serviceCurrency || ! $serviceCurrency->tree_accounting_id) {
            return false;
        }

        $serviceCostAccountId = $serviceCurrency->tree_accounting_id;

        // Get default currency and exchange rate
        $defaultCurrency = Currency::where('is_default', 1)->first();
        $currency = Currency::find($invoiceService->currency_id);
        $exchangeRate = $currency ? ($currency->is_default == 1 ? 1 : $currency->exchange_rate) : 1;

        // Get taxes from invoice_services_tax and calculate total tax amount
        $taxes = InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)
            ->where('currency_id', $invoiceService->currency_id)
            ->get();

        // Calculate total tax from invoice_services_tax (not from invoice_services.total_tax)
        $totalTaxAmount = (float) $taxes->sum('tax_amount');

        // Calculate net amount: grand_total (includes taxes) - total taxes = service cost
        $grandTotal = (float) ($invoiceService->grand_total ?? 0);
        $netAmount = max(0, $grandTotal - $totalTaxAmount);

        // Build transfers array
        $transfers = [];

        // 1. Service cost account (debit) - net amount without tax
        if ($netAmount > 0) {
            $transfers[] = [
                'tree_accounting_id' => $serviceCostAccountId,
                'currency_id' => $invoiceService->currency_id,
                'name' => 'تكلفة الخدمة - خدمة رقم '.$invoiceService->id,
                'debit' => round($netAmount * $exchangeRate, 2),
                'creditor' => 0,
                'currency_debit' => round($netAmount, 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ];
        }

        // 2. Tax accounts (debit) - from invoice_services_tax

        foreach ($taxes as $tax) {
            if ($tax->tax_rate_id && $tax->tax_amount > 0) {
                // Get tax account from tax_rate_mappings (type = sales)
                $taxMapping = TaxRateMappings::where('tax_rate_id', $tax->tax_rate_id)
                    ->where('currency_id', $invoiceService->currency_id)
                    ->where('type', 'sales')
                    ->first();

                if ($taxMapping && $taxMapping->tree_account_id) {
                    $transfers[] = [
                        'tree_accounting_id' => $taxMapping->tree_account_id,
                        'currency_id' => $invoiceService->currency_id,
                        'name' => 'ضريبة - خدمة رقم '.$invoiceService->id,
                        'debit' => round($tax->tax_amount * $exchangeRate, 2),
                        'creditor' => 0,
                        'currency_debit' => round($tax->tax_amount, 2),
                        'currency_creditor' => 0,
                        'cost_center_id' => null,
                        'currency_transfer_rate' => $exchangeRate,
                    ];
                }
            }
        }

        // 3. Supplier/Guide account (credit)
        $supplierAccountId = null;

        // For tour_reps: priority is guide_id first, then supplier_id
        if ($invoiceService->travel_tourism_type === 'tour_reps') {
            // Priority 1: If guide_id exists, use guide account from user_account_mappings
            if ($invoiceService->guide_id) {
                $guideMapping = \App\Models\UserAccountMapping::where('user_id', $invoiceService->guide_id)
                    ->where('currency_id', $invoiceService->currency_id)
                    ->first();

                if (! $guideMapping || ! $guideMapping->Supplier_account_id) {
                    // Cannot create constraint without guide supplier account mapping
                    return false;
                }

                $supplierAccountId = $guideMapping->Supplier_account_id;
            } elseif ($invoiceService->supplier_id) {
                // Priority 2: If no guide_id, use supplier_id from supplier_account_mappings
                $supplierMapping = SupplierAccountMappings::where('supplier_id', $invoiceService->supplier_id)
                    ->where('currency_id', $invoiceService->currency_id)
                    ->where('type', SupplierAccountMappingsEnum::START_BALANCE->value)
                    ->first();

                if (! $supplierMapping || ! $supplierMapping->tree_account_id) {
                    // Cannot create constraint without supplier account mapping
                    return false;
                }

                $supplierAccountId = $supplierMapping->tree_account_id;
            } else {
                // No guide_id and no supplier_id for tour_reps
                return false;
            }
        } else {
            // For other types: supplier_id is required
            if ($invoiceService->supplier_id) {
                $supplierMapping = SupplierAccountMappings::where('supplier_id', $invoiceService->supplier_id)
                    ->where('currency_id', $invoiceService->currency_id)
                    ->where('type', SupplierAccountMappingsEnum::START_BALANCE->value)
                    ->first();

                if (! $supplierMapping || ! $supplierMapping->tree_account_id) {
                    // Cannot create constraint without supplier account mapping
                    return false;
                }

                $supplierAccountId = $supplierMapping->tree_account_id;
            } else {
                // If no supplier_id, skip constraint creation
                return false;
            }
        }

        // Calculate total debit before adding credit
        $totalDebit = array_sum(array_column($transfers, 'debit'));
        $grandTotalInDefault = round($grandTotal * $exchangeRate, 2);

        // Add supplier/guide credit transfer
        if ($supplierAccountId) {
            $transfers[] = [
                'tree_accounting_id' => $supplierAccountId,
                'currency_id' => $invoiceService->currency_id,
                'name' => 'مورد - خدمة رقم '.$invoiceService->id,
                'debit' => 0,
                'creditor' => $grandTotalInDefault,
                'currency_debit' => 0,
                'currency_creditor' => round($grandTotal, 2),
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ];
        }

        // Calculate totals after adding credit
        $totalDebit = array_sum(array_column($transfers, 'debit'));
        $totalCreditor = array_sum(array_column($transfers, 'creditor'));

        // Validate accounting balance (debit must equal credit)
        $balanceDifference = abs($totalDebit - $totalCreditor);
        if ($balanceDifference > 0.02) {
            // Accounting constraint is not balanced
            return false;
        }

        // Adjust for minor rounding differences (if any)
        if ($balanceDifference > 0 && $balanceDifference <= 0.02) {
            $lastIndex = count($transfers) - 1;
            if ($totalDebit > $totalCreditor) {
                $transfers[$lastIndex]['creditor'] = round($transfers[$lastIndex]['creditor'] + $balanceDifference, 2);
            } else {
                $transfers[$lastIndex]['creditor'] = round($transfers[$lastIndex]['creditor'] - $balanceDifference, 2);
            }
            // Recalculate after adjustment
            $totalCreditor = array_sum(array_column($transfers, 'creditor'));
        }

        // Create constraint
        $constraintData = [
            'supplier_id' => $invoiceService->supplier_id,
            'profile_id' => $invoiceService->profile_id,
            'date' => date('Y-m-d'),
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'invoice_services_id' => $invoiceService->id,
            'name' => 'قيد استلام الخدمة من المورد',
            'description' => 'قيد استلام الخدمة - خدمة رقم '.$invoiceService->id,
            'active' => 1,
            'capture_exchange' => 'constraint',
            'creation_mode' => 'automatic',
            'currency_id' => $invoiceService->currency_id,
            'currency_transfer_rate' => $exchangeRate,
            'company_id' => $invoiceService->company_id ?? $invoiceService->profile->company_id ?? null,
            'transfers' => $transfers,
        ];

        // Store invoice_services_id in description or use model relationship
        // We'll use model_type and model_id to link constraint to invoice_services
        $constraintRepo = new \App\sys\Repository\Accounting\ConstraintRepository;
        $constraint = $constraintRepo->create($constraintData, 'yes');

        // Link constraint to invoice_services using transfers
        if ($constraint && $constraint->id) {
            // Update transfers to include invoice_services_id reference
            Transfer::where('constraint_id', $constraint->id)
                ->update(['invoice_id' => $invoiceService->invoice_id ?? null]);
        }

        return true;
    }

    /**
     * Find purchase invoice constraint for invoice_services
     */
    private function findPurchaseInvoiceConstraint(int $invoiceServicesId)
    {
        return Constraint::where('description', 'like', '%خدمة رقم '.$invoiceServicesId.'%')
            ->where('capture_exchange', 'constraint')
            ->where('creation_mode', 'automatic')
            ->first();
    }

    /**
     * Delete purchase invoice constraint for invoice_services
     */
    private function deletePurchaseInvoiceConstraint(int $invoiceServicesId)
    {
        // Find constraints linked to this invoice_services
        // We'll search by checking transfers that might reference this service
        // Since we don't have direct link, we'll search by description pattern
        $constraints = Constraint::where('description', 'like', '%خدمة رقم '.$invoiceServicesId.'%')
            ->where('capture_exchange', 'constraint')
            ->where('creation_mode', 'automatic')
            ->get();

        foreach ($constraints as $constraint) {
            // Delete transfers first
            Transfer::where('constraint_id', $constraint->id)->delete();
            // Delete constraint
            $constraint->delete();
        }

        return true;
    }
}
