<?php

namespace App\sys\Repository\Suppliers;

use App\Models\Accounting\TreeAccounting;
use App\Models\General\Service;
use App\Models\Suppliers\SupplierAccountMappings;
use App\Models\Suppliers\Suppliers;

class SupplierRepository
{
    private $columns = [
        'id' => 'id',
        'supplier_name' => 'supplier_name',
        'active' => 'supplier_active',
        'created_at' => 'created_at',
        'updated_at' => 'modified',
    ];

    private $supplier;

    public function __construct()
    {
        $this->supplier = new Suppliers;
    }

    public function getLang($id)
    {
        return $this->supplier->with('translations')->find($id);
    }

    public function findByIdOrFail($id)
    {
        return $this->supplier->with(
            'currentTranslation',
            'domains',
            'accounting',
            'accommodations',
            'services',
            'company.currentTranslation'
        )->find($id);
    }

    public function active()
    {
        return $this->supplier->with('currentTranslation', 'company.currentTranslation')->where('supplier_active', 1)->get();
    }

    public function del(array $ids)
    {
        return $this->supplier->whereIn('id', $ids)->delete();
    }

    public function add($data)
    {
        $this->supplier->supplier_active = $data['active'] ?? 1;
        $this->supplier->supplier_name = $data['supplier_name'];
        $this->supplier->supplier_vat = $data['supplier_vat'] ?? null;
        $this->supplier->supplier_address = $data['supplier_address'] ?? null;
        $this->supplier->supplier_zip = $data['supplier_zip'] ?? null;
        $this->supplier->supplier_email = $data['supplier_email'] ?? null;
        $this->supplier->supplier_phone = $data['supplier_phone'] ?? null;
        $this->supplier->supplier_fax = $data['supplier_fax'] ?? null;
        $this->supplier->company_id = $data['company_id'] ?? null;
        $this->supplier->supplier_social_ld = $data['supplier_social_ld'] ?? null;
        $this->supplier->supplier_social_fb = $data['supplier_social_fb'] ?? null;
        $this->supplier->supplier_social_tw = $data['supplier_social_tw'] ?? null;
        $this->supplier->supplier_contact_person = $data['supplier_contact_person'] ?? null;
        $this->supplier->supplier_contact_phone = $data['supplier_contact_phone'] ?? null;
        $this->supplier->supplier_contact_email = $data['supplier_contact_email'] ?? null;
        $this->supplier->cash_deal_type = $data['payment_method'] ?? 'pay_on_booking';
        $this->supplier->pay_on_booking = $data['pay_on_booking'] ?? null;
        $this->supplier->save();
        $this->supplier->services()->sync($data['services']);
        $this->supplier->accommodations()->sync($data['accommodations']);
        $this->supplier->domains()->sync($data['domains']);
        $this->saveAccounting($data, $this->supplier->id, 'accounting');
        $this->saveAccounting($data, $this->supplier->id, 'payment_method_accounting');

        return $this->supplier->load('accounting');
    }

    public function updated($data)
    {
        $supplier = $this->supplier->find($data['id']);
        $supplier->supplier_active = $data['active'] ?? $supplier->supplier_active;
        $supplier->supplier_name = $data['supplier_name'] ?? $supplier->supplier_name;
        $supplier->supplier_vat = $data['supplier_vat'] ?? $supplier->supplier_vat;
        $supplier->company_id = $data['company_id'] ?? $supplier->company_id;
        $supplier->supplier_address = $data['supplier_address'] ?? $supplier->supplier_address;
        $supplier->supplier_zip = $data['supplier_zip'] ?? $supplier->supplier_zip;
        $supplier->supplier_email = $data['supplier_email'] ?? $supplier->supplier_email;
        $supplier->supplier_phone = $data['supplier_phone'] ?? $supplier->supplier_phone;
        $supplier->supplier_fax = $data['supplier_fax'] ?? $supplier->supplier_fax;
        $supplier->supplier_social_ld = $data['supplier_social_ld'] ?? $supplier->supplier_social_ld;
        $supplier->supplier_social_fb = $data['supplier_social_fb'] ?? $supplier->supplier_social_fb;
        $supplier->supplier_social_tw = $data['supplier_social_tw'] ?? $supplier->supplier_social_tw;
        $supplier->supplier_contact_person = $data['supplier_contact_person'] ?? $supplier->supplier_contact_person;
        $supplier->supplier_contact_phone = $data['supplier_contact_phone'] ?? $supplier->supplier_contact_phone;
        $supplier->supplier_contact_email = $data['supplier_contact_email'] ?? $supplier->supplier_contact_email;
        $supplier->cash_deal_type = $data['payment_method'] ?? $supplier->cash_deal_type;
        $supplier->pay_on_booking = $data['pay_on_booking'] ?? $supplier->pay_on_booking;
        $supplier->domain_id = $data['domain_id'] ?? $supplier->domain_id;
        $supplier->save();
        if (isset($data['accommodations']) && ! empty($data['accommodations'])) {
            $supplier->accommodations()->sync($data['accommodations']);
        }
        if (isset($data['services']) && ! empty($data['services'])) {
            $supplier->services()->sync($data['services']);
        }
        if (isset($data['domains']) && ! empty($data['domains'])) {
            $supplier->domains()->sync($data['domains']);
        }
        // $this->updatedAccounting($data, $supplier->id, 'accounting');
        $this->updatedAccounting($data, $supplier->id, 'payment_method_accounting');

        return $supplier->load('accounting');
    }

    public function saveAccounting($data, $id, $name)
    {
        foreach ($data[$name] ?? [] as $value) {
            $this->addAccounting($value, $id);
        }
    }

    private function updatedAccounting($data, $id, $name)
    {
        foreach ($data[$name] ?? [] as $value) {
            if (SupplierAccountMappings::where(['supplier_id' => $id, 'type' => $value['input_name'], 'currency_id' => $value['currency_id']])->exists()) {
                $this->updatethisAccount($value, $id);
            } else {
                $this->addAccounting($value, $id);
            }
        }
    }

    public function updatethisAccount($data, $id)
    {
        $account = SupplierAccountMappings::where(['supplier_id' => $id, 'type' => $data['input_name'], 'currency_id' => $data['currency_id']])->first();
        $account->tree_account_id = $data['tree_account_id'] ?? $account->tree_account_id;
        $account->balance = $data['balance'] ?? $account->balance;
        $account->account_type = $data['account_type'] ?? $account->account_type;
        $account->save();
    }

    public function addAccounting($data, $id)
    {
        $account = new SupplierAccountMappings;
        $account->supplier_id = $id;
        $account->currency_id = $data['currency_id'];
        $account->balance = $data['balance'] ?? 0;
        $account->tree_account_id = $data['tree_account_id'] ?? null;
        $account->type = $data['input_name'];
        $account->account_type = $data['account_type'] ?? 'credit';
        $account->save();
    }

    public function get()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);
        $query = $this->supplier->query();
        $query->when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('supplier_name', 'LIKE', "%$name%");
            });
        });
        $query->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'supplier_name') {
                $query->leftJoin('su_supplier_translation', function ($join) {
                    $join->on('su_supplier.id', '=', 'su_supplier_translation.supplier_id')
                        ->where('su_supplier_translation.lang_id', app('lang_id'));
                })->orderBy('su_supplier_translation.supplier_name', $order)->select('area.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        });

        return $query->with('currentTranslation', 'company.currentTranslation')->paginate($limit);
    }

    private function SupplierServices($data) {}

    public function supplierByServiceAndCurrency($serviceId, $currencyId)
    {
        return Service::where('id', $serviceId)
            ->with(['suppliers' => function ($query) use ($currencyId) {
                $query->whereHas('accounting', function ($q) use ($currencyId) {
                    $q->where('currency_id', $currencyId);
                })->with(['currentTranslation', 'company.currentTranslation', 'accounting' => function ($q) use ($currencyId) {
                    $q->where('currency_id', $currencyId);
                }]);
            }])->first();
    }

    public function suppliersByAccommodationId(int $accommodationId)
    {
        return $this->supplier
            ->whereHas('accommodations', function ($q) use ($accommodationId) {
                $q->where('accommodation_id', $accommodationId);
            })
            ->with([
                'currentTranslation',
                'company.currentTranslation',
                'accounting.currency.currentTranslation',
            ])
            ->get();
    }

    public function getSupplierAccountMappings(int $supplierId, string $type)
    {
        return SupplierAccountMappings::where('supplier_id', $supplierId)
            ->where('type', $type)
            ->get();
    }

    public function getTreeAccountsWithTransfers(array $treeAccountIds)
    {
        return TreeAccounting::whereIn('id', $treeAccountIds)
            ->with([
                'currentTranslation',
                'transfers' => function ($query) {
                    // Get all transfers, can add date filters later if needed
                },
                'transfers.constraint',
                'transfers.costCenter',
                'transfers.currency.currentTranslation',
                'transfers.treeAccounting',
            ])->get();
    }
}
