<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\DomainRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class DomainService extends Services
{
    protected DomainRepository $domainRepository;

    public function __construct(DomainRepository $domainRepository)
    {
        $this->domainRepository = $domainRepository;
    }

    public function getPaginatedDomains()
    {
        return $this->domainRepository->getPaginated();
    }

    public function getDomainById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:domains,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->domainRepository->findByIdOrFail($id);
    }

    public function createDomain($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'tax_rate_id' => ['required', 'integer', 'exists:taxs_rate,id'],
            'type' => ['required', 'string', 'in:sales,returns'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $domain = $this->domainRepository->create($request);
        $this->syncTranslations($domain);

        return $domain;
    }

    public function updateDomain($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:domains,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'tax_rate_id' => ['sometimes', 'nullable', 'integer', 'exists:taxs_rate,id'],
            'type' => ['sometimes', 'string', 'in:sales,returns'],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $domain = $this->domainRepository->findByIdOrFail($request['id']);

        return $this->domainRepository->update($domain, $request);
    }

    public function deleteDomain(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:domains,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $domain = $this->domainRepository->findByIdOrFail($id);

        return $this->domainRepository->delete($domain);
    }

    public function del(array $ids)
    {
        return $this->domainRepository->del($ids);
    }

    public function getActive()
    {
        return $this->domainRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:domains,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $domain = $this->domainRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($domain, true);
    }

    public function getByTaxRateId(int $taxRateId)
    {
        $rules = [
            'tax_rate_id' => ['required', 'integer', 'exists:taxs_rate,id'],
        ];
        $validator = Validator::make(['tax_rate_id' => $taxRateId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->domainRepository->getByTaxRateId($taxRateId);
    }

    public function getAllChanges($domain)
    {
        return $this->domainRepository->getAllChanges($domain);
    }
}
