<?php

namespace App\sys\Services\General;

use App\Models\Accounting\TreeAccounting;
use App\Rules\MatchCurrenciesCount;
use App\sys\Repository\General\ServiceRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class ServiceService extends Services
{
    protected ServiceRepository $serviceRepository;

    public function __construct(ServiceRepository $serviceRepository)
    {
        $this->serviceRepository = $serviceRepository;
    }

    public function getPaginatedCountries()
    {
        return $this->serviceRepository->getPaginated();
    }

    public function getServiceById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:services,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->serviceRepository->findByIdOrFail($id);
    }

    public function createService($request)
    {
        $rules = [
            'type' => ['required', 'in:additional_services,tour_guide,daily_transportation,single_transportation,tour_reps,dining_entertainment,attractions_temples,flights'],
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('services', 'title')->where(
                    fn ($query) => $query->where('type', request('type'))
                ),
            ],
            'is_fixed_price' => ['required', 'boolean'],
            'item_code' => ['required', 'string', 'max:255'],
            'unit_type' => ['required', 'string', 'max:50'],
            'active' => ['required', 'boolean'],
            'supplier_ids' => ['nullable', 'array'],
            'supplier_ids.*' => ['integer', 'exists:su_supplier,id'],
            'user_ids' => ['nullable', 'array'],
            'user_ids.*' => ['integer', 'exists:users,id'],
            'services_currencies' => ['required', 'array', new MatchCurrenciesCount],
            'services_currencies.*.currency_id' => [
                'required',
                'integer',
                'distinct',
                Rule::exists('currencies', 'id')->where(function ($query) {
                    $query->where('status', 1)
                        ->whereNull('deleted_at');
                }),
            ],
            'services_currencies.*.tree_accounting_id' => [
                'required_with:services_currencies',
                'integer',
                'exists:tree_accounting,id',
                function ($attribute, $value, $fail) use ($request) {
                    $index = explode('.', $attribute)[1];
                    $currencyId = $request['services_currencies'][$index]['currency_id'] ?? null;
                    if ($currencyId) {
                        $treeAccounting = TreeAccounting::find($value);
                        if ($treeAccounting && $treeAccounting->currency_id != $currencyId) {
                            $fail("The tree_accounting_id at {$attribute} must have the same currency as the currency_id.");
                        }
                    }
                },
            ],
            'services_currencies.*.tree_income_accounting_id' => [
                'required_with:services_currencies',
                'integer',
                'exists:tree_accounting,id',
                function ($attribute, $value, $fail) use ($request) {
                    $index = explode('.', $attribute)[1];
                    $currencyId = $request['services_currencies'][$index]['currency_id'] ?? null;
                    if ($currencyId) {
                        $treeAccounting = TreeAccounting::find($value);
                        if ($treeAccounting && $treeAccounting->currency_id != $currencyId) {
                            $fail("The tree_income_accounting_id at {$attribute} must have the same currency as the currency_id.");
                        }
                    }
                },
            ],
            'services_currencies.*.adult_price' => 'nullable|numeric|min:0',
            'services_currencies.*.child_price' => 'nullable|numeric|min:0',
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $service = $this->serviceRepository->create($request);
        $this->syncTranslations($service);

        return $service;
    }

    public function updateService($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:services,id'],
            'type' => ['sometimes', 'in:additional_services,tour_guide,daily_transportation,single_transportation,tour_reps,dining_entertainment,attractions_temples,flights'],
            'name' => [
                'sometimes',
                'string',
                'max:255',
                Rule::unique('services', 'title')
                    ->when(
                        isset($request['type']),
                        fn ($rule) => $rule->where(
                            fn ($query) => $query->where('type', $request['type'])
                        )
                    )
                    ->ignore($request['id']),

            ],
            'is_fixed_price' => ['sometimes', 'boolean'],
            'item_code' => ['sometimes', 'string', 'max:255'],
            'unit_type' => ['sometimes', 'string', 'max:50'],
            'active' => ['sometimes', 'boolean'],

            'supplier_ids' => ['sometimes', 'array'],
            'supplier_ids.*' => ['integer', 'exists:su_supplier,id'],

            'user_ids' => ['sometimes', 'array'],
            'user_ids.*' => ['integer', 'exists:users,id'],

            'services_currencies' => ['sometimes', 'array', new MatchCurrenciesCount],

            'services_currencies.*.currency_id' => [
                'required_with:services_currencies',
                'integer',
                'distinct',
                Rule::exists('currencies', 'id')->where(function ($query) {
                    $query->where('status', 1)
                        ->whereNull('deleted_at');
                }),
            ],
            'services_currencies.*.tree_accounting_id' => [
                'required_with:services_currencies',
                'integer',
                'exists:tree_accounting,id',
                function ($attribute, $value, $fail) use ($request) {
                    $index = explode('.', $attribute)[1];
                    $currencyId = $request['services_currencies'][$index]['currency_id'] ?? null;
                    if ($currencyId) {
                        $treeAccounting = TreeAccounting::find($value);
                        if ($treeAccounting && $treeAccounting->currency_id != $currencyId) {
                            $fail("The tree_accounting_id at {$attribute} must have the same currency as the currency_id.");
                        }
                    }
                },
            ],
            'services_currencies.*.tree_income_accounting_id' => [
                'required_with:services_currencies',
                'integer',
                'exists:tree_accounting,id',
                function ($attribute, $value, $fail) use ($request) {
                    $index = explode('.', $attribute)[1];
                    $currencyId = $request['services_currencies'][$index]['currency_id'] ?? null;
                    if ($currencyId) {
                        $treeAccounting = TreeAccounting::find($value);
                        if ($treeAccounting && $treeAccounting->currency_id != $currencyId) {
                            $fail("The tree_income_accounting_id at {$attribute} must have the same currency as the currency_id.");
                        }
                    }
                },
            ],
            'services_currencies.*.adult_price' => 'nullable|numeric|min:0',
            'services_currencies.*.child_price' => 'nullable|numeric|min:0',
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $service = $this->serviceRepository->findByIdOrFail($request['id']);

        return $this->serviceRepository->update($service, $request);
    }

    public function getActive()
    {
        return $this->serviceRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:services,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $service = $this->serviceRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($service, true);
    }

    public function getByType(string $type)
    {
        $rules = [
            'type' => ['required', 'in:additional_services,tour_guide,daily_transportation,single_transportation,tour_reps,dining_entertainment,attractions_temples,flights'],
        ];

        $validator = Validator::make(['type' => $type], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->serviceRepository->getByType($type);
    }

    public function del(array $ids)
    {
        return $this->serviceRepository->del($ids);
    }

    public function getCurrencyWithTree($currencyId = null)
    {
        $services = $this->serviceRepository->getActive();
        $result = [];

        foreach ($services as $service) {
            $currencies = $service->getCurrencyWithTree($currencyId);
            if ($currencies->count() > 0) {
                $result[] = [
                    'service_id' => $service->id,
                    'service_title' => $service->title,
                    'currencies' => $currencies,
                ];
            }
        }

        return $result;
    }

    public function getServicePrice($serviceId, $currencyId)
    {
        return $this->serviceRepository->getPricesByServiceId($serviceId, $currencyId);
    }
}
