<?php

namespace App\sys\Services\Suppliers;

use App\sys\Repository\Suppliers\SupplierTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class SupplierTranslationService extends Services
{
    private SupplierTranslationRepository $supplierTranslationRepository;

    public function __construct()
    {
        $this->supplierTranslationRepository = new SupplierTranslationRepository;
    }

    public function save(array $data, int $supplierId)
    {
        $rules = ['supplier_id' => 'required|integer|exists:su_supplier,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:su_supplier_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('su_supplier_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('supplier_id', $supplierId))
                    ->ignore($rowId),
            ];
            $rules["$index.supplier_name"] = 'nullable|string|max:255';
            $rules["$index.supplier_address"] = 'nullable|string|max:500';

            if ($languageId) {
                $key = $supplierId.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language duplicated for the same Supplier.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['supplier_id' => $supplierId]), $rules);

        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->supplierTranslationRepository->save($data, $supplierId);
    }

    public function get(int $supplierId)
    {
        return $this->supplierTranslationRepository->get($supplierId);
    }

    public function del(int $id)
    {
        $rules = [
            'id' => 'required|integer|exists:su_supplier_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->supplierTranslationRepository->del($id);
    }
}
