<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\General\TaxRate\TaxRateLangResource;
use App\Http\Resources\General\TaxRateHistoryResource;
use App\Http\Resources\TaxRateCollection;
use App\Http\Resources\TaxRateResource;
use App\sys\ApiResponse;
use App\sys\Services\General\TaxRateService;
use App\sys\Services\General\TaxRateTranslationService;
use Illuminate\Http\Request;

class TaxRateController extends Controller
{
    use ApiResponse;

    protected TaxRateService $taxRateService;

    protected TaxRateTranslationService $taxRateTranslationService;

    public function __construct(TaxRateService $taxRateService)
    {
        $this->taxRateService = $taxRateService;
        $this->taxRateTranslationService = new TaxRateTranslationService;
    }

    public function index()
    {
        $taxRates = $this->taxRateService->getPaginatedTaxRates();

        return $this->apiResponse(200, 'success', null, new TaxRateCollection($taxRates));
    }

    public function show(int $id)
    {
        $taxRate = $this->taxRateService->getTaxRateById($id);
        if (! $taxRate) {
            $errors = $this->taxRateService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new TaxRateResource($taxRate));
    }

    public function store(Request $request)
    {
        $response = $this->taxRateService->createTaxRate($request->all());

        if (! $response) {
            $errors = $this->taxRateService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new TaxRateResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->taxRateService->updateTaxRate($request->all());
        if (! $response) {
            $errors = $this->taxRateService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new TaxRateResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->taxRateService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Tax Rates deleted successfully', null, null);
        }
        $errors = $this->taxRateService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->taxRateService->getActive();
        $collect = new TaxRateCollection($data);

        return $this->apiResponse(200, 'Active Tax Rates retrieved successfully', null, $collect);
    }

    public function getCurrencyWithTree()
    {
        $data = $this->taxRateService->getCurrencyWithtree();

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function getByCurrency(int $currencyId)
    {
        $data = $this->taxRateService->getByCurrency($currencyId);
        if (! $data) {
            $errors = $this->taxRateService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }
        $tax = new TaxRateCollection($data);

        return $this->apiResponse(200, 'success', null, $tax);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->taxRateTranslationService->save($request->all(), $id)) {
            $errors = $this->taxRateTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->taxRateService->getByIdWithTranslation($id)) {
            $collection = new TaxRateLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->taxRateService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->taxRateTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->taxRateTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $taxRate = $this->taxRateService->getTaxRateById($id);
        if (! $taxRate) {
            $errors = $this->taxRateService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->taxRateService->getAllChanges($taxRate);

        return $this->apiResponse(200, 'success', null, TaxRateHistoryResource::collection($changes));
    }
}
