<?php

namespace App\Observers;

use App\Models\Profile\Profile;
use App\sys\Enums\GroupEnum;
use App\sys\Services\Notification\NotificationService;

class ProfileObserver
{
    protected NotificationService $notificationService;

    /**
     * المجموعات المستهدفة بإشعارات Profile
     */
    protected const TARGET_GROUPS = [
        GroupEnum::ADMIN,
        GroupEnum::SALES_MANAGER,
    ];

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Handle the Profile "created" event.
     */
    public function created(Profile $profile): void
    {
        // إرسال إشعار لجميع الـ Admins و Sales Managers عند إنشاء Profile جديد
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'Profile',
                title: 'تم إنشاء Profile جديد',
                message: "تم إنشاء Profile رقم {$profile->profile_number} بنجاح",
                modelName: 'Profile',
                modelId: $profile->id,
                data: [
                    'profile_number' => $profile->profile_number,
                    'customer_id' => $profile->customer_id,
                    'user_id' => $profile->user_id,
                    'action' => 'created',
                ],
                pushChannel: 'group'
            );
        }
    }

    /**
     * Handle the Profile "updated" event.
     */
    public function updated(Profile $profile): void
    {
        // إرسال إشعار لجميع الـ Admins و Sales Managers عند تحديث Profile
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'Profile',
                title: 'تم تحديث Profile',
                message: "تم تحديث Profile رقم {$profile->profile_number}",
                modelName: 'Profile',
                modelId: $profile->id,
                data: [
                    'profile_number' => $profile->profile_number,
                    'action' => 'updated',
                ],
                pushChannel: 'group'
            );
        }
    }

    /**
     * Handle the Profile "deleted" event.
     */
    public function deleted(Profile $profile): void
    {
        // إرسال إشعار لجميع الـ Admins و Sales Managers عند حذف Profile
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'Profile',
                title: 'تم حذف Profile',
                message: "تم حذف Profile رقم {$profile->profile_number}",
                modelName: 'Profile',
                modelId: $profile->id,
                data: [
                    'profile_number' => $profile->profile_number,
                    'action' => 'deleted',
                ],
                pushChannel: 'group'
            );
        }
    }

    /**
     * Handle the Profile "restored" event.
     */
    public function restored(Profile $profile): void
    {
        // إرسال إشعار لجميع الـ Admins و Sales Managers عند استعادة Profile
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'Profile',
                title: 'تم استعادة Profile',
                message: "تم استعادة Profile رقم {$profile->profile_number}",
                modelName: 'Profile',
                modelId: $profile->id,
                data: [
                    'profile_number' => $profile->profile_number,
                    'action' => 'restored',
                ],
                pushChannel: 'group'
            );
        }
    }

    /**
     * Handle the Profile "force deleted" event.
     */
    public function forceDeleted(Profile $profile): void
    {
        // إرسال إشعار لجميع الـ Admins و Sales Managers عند الحذف النهائي لـ Profile
        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'Profile',
                title: 'تم الحذف النهائي لـ Profile',
                message: "تم الحذف النهائي لـ Profile رقم {$profile->profile_number}",
                modelName: 'Profile',
                modelId: $profile->id,
                data: [
                    'profile_number' => $profile->profile_number,
                    'action' => 'force_deleted',
                ],
                pushChannel: 'group'
            );
        }
    }
}
