<?php

namespace App\sys\Repository\General;

use App\Models\General\Category;

class CategoryRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'title',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $title = request('name', null);
        $limit = request('limit', 15);

        return Category::when($title, function ($q, $title) {
            $q->whereHas('translations', function ($q) use ($title) {
                $q->where('name', 'LIKE', "%$title%")->where('lang_id', app('lang_id'));
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('category_translation', function ($join) {
                    $join->on('categories.id', '=', 'category_translation.category_id')
                        ->where('category_translation.lang_id', app('lang_id'));
                })->orderBy('category_translation.title', $order)->select('categories.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail($id)
    {
        return Category::with('currentTranslation')->find($id);
    }

    public function create(array $data)
    {
        $category = new Category;
        $category->title = $data['name'];
        $category->is_active = $data['active'];
        $category->save();

        return $category;
    }

    public function update(Category $category, array $data)
    {
        $category->title = $data['name'] ?? $category->title;
        $category->is_active = $data['active'] ?? $category->is_active;
        $category->save();

        return $category;
    }

    public function getActive()
    {
        return Category::with('currentTranslation')->whereNull('deleted_at')->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Category::with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return Category::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($category)
    {
        return $category->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
