<?php

namespace App\sys\Repository\General;

use App\Models\General\TransportationModes;

class TransportationModesRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function index()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return TransportationModes::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ge_transportation_modes_translation', function ($join) {
                    $join->on('ge_transportation_modes.id', '=', 'ge_transportation_modes_translation.tranortation_id')
                        ->where('ge_transportation_modes_translation.lang_id', app('lang_id'));
                })->orderBy('ge_transportation_modes_translation.name', $order)->select('ge_transportation_modes.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function getById($id)
    {
        return TransportationModes::find($id);
    }

    public function create(array $data)
    {
        $item = new TransportationModes;
        $item->name = $data['name'];
        $item->is_active = $data['active'] ?? 1;
        $item->save();

        return $item;
    }

    public function update($data)
    {
        $item = TransportationModes::find($data['id']);
        $item->name = $data['name'] ?? $item->name;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->save();

        return $item;
    }

    public function getActive()
    {
        return TransportationModes::with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return TransportationModes::with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return TransportationModes::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($transportationModes)
    {
        return $transportationModes->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
