<?php

namespace App\sys\Services\Accommodation;

use App\sys\Repository\Accommodation\AccommodationsTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class AccommodationsTranslationService extends Services
{
    private $accommodationsTranslation;

    public function __construct()
    {
        $this->accommodationsTranslation = new AccommodationsTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['accommodation_id' => 'required|integer|exists:ac_accommodations,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['lang_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:ac_accommodations_translation,id';
            $rules["$index.lang_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('ac_accommodations_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('accommodation_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';
            $rules["$index.description"] = 'nullable|string';
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.lang_id"] = 'Same language distorted with the same Accommodation.';
                } else {
                    $combinations[] = $key;
                }
            }
        }
        $validator = Validator::make(array_merge($data, ['accommodation_id' => $id]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->accommodationsTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->accommodationsTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:ac_accommodations_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->accommodationsTranslation->del($id);
    }
}
