<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\CostCenterRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CostCenterService extends Services
{
    protected CostCenterRepository $costCenterRepository;

    public function __construct(CostCenterRepository $costCenterRepository)
    {
        $this->costCenterRepository = $costCenterRepository;
    }

    public function getById(int $id)
    {
        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:cost_centers,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->costCenterRepository->findById($id);
    }

    public function create(array $request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:200'],
            'cost_center_id' => ['nullable', 'integer', 'exists:cost_centers,id'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // توليد serial number تلقائياً - لا نأخذه من الفرونت
        $parentId = $request['cost_center_id'] ?? null;
        $serialData = $this->costCenterRepository->generateNextSerial($parentId);

        if (isset($serialData['error'])) {
            $this->setError(['serial' => [$serialData['error']]]);

            return false;
        }

        // إضافة number و full_number للـ request - دائماً من التوليد التلقائي
        $request['number'] = $serialData['number'];
        $request['full_number'] = $serialData['full_number'];

        return $this->costCenterRepository->create($request);
    }

    public function update(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:cost_centers,id'],
            'name' => ['sometimes', 'string', 'max:200'],
            'cost_center_id' => ['sometimes', 'nullable', 'integer', 'exists:cost_centers,id'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $costCenter = $this->getById($request['id']);
        if (! $costCenter) {
            return false;
        }

        // فقط إذا تم تغيير cost_center_id، نعيد توليد serial number
        // غير كدا مش هنعدل number و full_number
        if (isset($request['cost_center_id']) && $request['cost_center_id'] != $costCenter->cost_center_id) {
            $parentId = $request['cost_center_id'];

            // استثناء الـ cost center الحالي من البحث (لأنه قد يكون له أطفال)
            // هات آخر رقم تحت نفس الأب عدا الـ cost center الحالي
            $last = \App\Models\Accounting\CostCenter::where(function ($query) use ($parentId) {
                if (! $parentId || $parentId == 0) {
                    $query->whereNull('cost_center_id')
                        ->orWhere('cost_center_id', 0);
                } else {
                    $query->where('cost_center_id', $parentId);
                }
            })
                ->where('id', '!=', $costCenter->id)
                ->orderBy('number', 'desc')
                ->first();

            $nextNumber = $last ? ($last->number + 1) : 1;

            // بناء full_number
            if (! $parentId || $parentId == 0) {
                // تحقق من عدم تجاوز 9 في المستوى الأولي
                if ($nextNumber > 9) {
                    $this->setError(['cost_center_id' => ['تم الوصول إلى الحد الأقصى للمستوى الأولي (9)']]);

                    return false;
                }
                $fullNumber = (string) $nextNumber;
            } else {
                $parent = \App\Models\Accounting\CostCenter::find($parentId);
                if (! $parent) {
                    $this->setError(['cost_center_id' => ['Parent cost center not found']]);

                    return false;
                }
                $fullNumber = $parent->full_number.'-'.$nextNumber;
            }

            $request['number'] = $nextNumber;
            $request['full_number'] = $fullNumber;
        } else {
            // لو مش غير cost_center_id، مش هنعدل number و full_number
            // احذفهم من request لو موجودين
            unset($request['number']);
            unset($request['full_number']);
        }

        return $this->costCenterRepository->update($costCenter, $request);
    }

    public function del(array $ids)
    {
        $validator = Validator::make(['ids' => $ids], [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:cost_centers,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->costCenterRepository->del($ids);
    }

    public function getPaginated()
    {
        return $this->costCenterRepository->getPaginated();
    }

    /**
     * Get next serial number for a given parent
     *
     * @param  int|null  $parentId
     * @return array|false
     */
    public function getNextSerial($parentId = null)
    {
        $parentId = $parentId == 0 ? null : $parentId;

        if ($parentId !== null) {
            $validator = Validator::make(['cost_center_id' => $parentId], [
                'cost_center_id' => ['required', 'integer', 'exists:cost_centers,id'],
            ]);
            if ($validator->fails()) {
                $this->setError($validator->errors());

                return false;
            }
        }

        $serialData = $this->costCenterRepository->generateNextSerial($parentId);

        if (isset($serialData['error'])) {
            $this->setError(['serial' => [$serialData['error']]]);

            return false;
        }

        return $serialData;
    }

    public function getAll()
    {
        return $this->costCenterRepository->getAll();
    }

    public function getParents()
    {
        return $this->costCenterRepository->getAllParent();
    }

    public function getByParent($parentId)
    {
        $validator = Validator::make(['parent_id' => $parentId], [
            'parent_id' => ['required', 'integer'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->costCenterRepository->getbyparent($parentId);
    }
}
