<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CategoryRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CategoryService extends Services
{
    protected $categoryRepository;

    public function __construct()
    {
        $this->categoryRepository = new CategoryRepository;
    }

    public function getPaginatedCategories()
    {
        return $this->categoryRepository->getPaginated();
    }

    public function getCategoryById($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:categories,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->categoryRepository->findByIdOrFail($id);
    }

    public function createCategory($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:300', 'unique:categories,title'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $category = $this->categoryRepository->create($request);
        $this->syncTranslations($category);

        return $category;
    }

    public function updateCategory($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:categories,id'],
            'name' => ['sometimes', 'string', 'max:300'],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $category = $this->categoryRepository->findByIdOrFail($request['id']);

        return $this->categoryRepository->update($category, $request);
    }

    public function getActive()
    {
        return $this->categoryRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:categories,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $category = $this->categoryRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($category, true);
    }

    public function del(array $ids)
    {
        return $this->categoryRepository->del($ids);
    }

    public function getAllChanges($category)
    {
        return $this->categoryRepository->getAllChanges($category);
    }
}
