<?php

namespace App\sys\Services\Notification;

use App\Models\User;
use App\Notifications\GenericNotification;
use App\sys\Repository\Notification\NotificationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class NotificationService extends Services
{
    protected NotificationRepository $notificationRepository;

    public function __construct(NotificationRepository $notificationRepository)
    {
        $this->notificationRepository = $notificationRepository;
    }

    /**
     * الوظيفة الواحدة - تحفظ وترسل push للإشعار
     *
     * @param  array  $data  البيانات المطلوبة:
     *                       - user_id: ID المستخدم (للإشعارات الخاصة)
     *                       - channel: اسم القناة (للإشعارات العامة - public channel)
     *                       - type: نوع الإشعار (invoice_update, user_action, system_event)
     *                       - title: عنوان الإشعار
     *                       - message: نص الإشعار
     *                       - model_name: اسم الموديل (اختياري)
     *                       - model_id: ID الموديل (اختياري)
     *                       - sender_by: user_id المستخدم الذي قام بالإجراء (اختياري)
     *                       - data: بيانات إضافية (اختياري)
     *                       - expires_at: تاريخ انتهاء الإشعار (اختياري)
     * @return bool|array
     */
    public function sendNotification(array $data)
    {
        // Validation
        $rules = [
            'type' => 'required|string|in:invoice_update,user_action,system_event',
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'user_id' => 'nullable|integer|exists:users,id',
            'channel' => 'nullable|string|max:255',
            'model_name' => 'nullable|string|max:255',
            'model_id' => 'nullable|integer',
            'sender_by' => 'nullable|integer|exists:users,id',
            'data' => 'nullable|array',
            'expires_at' => 'nullable|date',
        ];

        // يجب أن يكون إما user_id أو channel
        if (empty($data['user_id']) && empty($data['channel'])) {
            $this->setError(['notification' => ['Either user_id or channel must be provided']]);

            return false;
        }

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $senderBy = $data['sender_by'] ?? auth()->id();

        // إنشاء الإشعار (الـ message الأصلي يبقى كما هو في قاعدة البيانات)
        $notification = new GenericNotification(
            title: $data['title'],
            message: $data['message'],
            type: $data['type'],
            modelName: $data['model_name'] ?? null,
            modelId: $data['model_id'] ?? null,
            senderBy: $senderBy,
            data: $data['data'] ?? [],
            channel: $data['channel'] ?? null,
            expiresAt: isset($data['expires_at']) ? new \DateTime($data['expires_at']) : null
        );

        // إرسال الإشعار
        if (! empty($data['user_id'])) {
            // إشعار خاص بمستخدم - لا نرسل للمرسل نفسه
            if ($data['user_id'] == $senderBy) {
                return true; // لا نرسل إشعار للمرسل نفسه
            }

            $user = User::find($data['user_id']);
            if (! $user) {
                $this->setError(['user_id' => ['User not found']]);

                return false;
            }
            $user->notify($notification);
        } else {
            // إشعار على قناة عامة (public channel)
            event(new \App\Events\PublicNotificationEvent(
                title: $data['title'],
                message: $data['message'],
                type: $data['type'],
                modelName: $data['model_name'] ?? null,
                modelId: $data['model_id'] ?? null,
                senderBy: $senderBy,
                data: $data['data'] ?? [],
                channel: $data['channel']
            ));
        }

        // حفظ expires_at في قاعدة البيانات إذا كان موجود
        if (isset($data['expires_at']) && ! empty($data['user_id'])) {
            $latestNotification = $user->notifications()->latest()->first();
            if ($latestNotification) {
                $latestNotification->update(['expires_at' => $data['expires_at']]);
            }
        }

        return true;
    }

    /**
     * Get user notifications
     */
    public function getUserNotifications(int $userId, int $limit = 15, bool $unreadOnly = false)
    {
        return $this->notificationRepository->getUserNotifications($userId, $limit, $unreadOnly);
    }

    /**
     * Get unread count
     */
    public function getUnreadCount(int $userId): int
    {
        return $this->notificationRepository->getUnreadCount($userId);
    }

    /**
     * Mark as read
     */
    public function markAsRead(string $notificationId, int $userId): bool
    {
        return $this->notificationRepository->markAsRead($notificationId, $userId);
    }

    /**
     * Mark all as read
     */
    public function markAllAsRead(int $userId): int
    {
        return $this->notificationRepository->markAllAsRead($userId);
    }

    /**
     * Delete notification
     */
    public function delete(string $notificationId, int $userId): bool
    {
        return $this->notificationRepository->delete($notificationId, $userId);
    }

    /**
     * Get single notification
     */
    public function find(string $notificationId, int $userId)
    {
        return $this->notificationRepository->find($notificationId, $userId);
    }

    /**
     * إرسال إشعار لكل المستخدمين في Group
     * يحفظ إشعار منفصل في DB لكل user + يرسل push على Group channel
     *
     * @param  int  $groupId  Group ID (مثلاً GroupEnum::ADMIN->value)
     * @param  string  $type  نوع الإشعار
     * @param  string  $title  عنوان الإشعار
     * @param  string  $message  نص الإشعار
     * @param  string|null  $modelName  اسم الموديل
     * @param  int|null  $modelId  ID الموديل
     * @param  array  $data  بيانات إضافية
     * @param  string  $pushChannel  نوع الـ push: 'group' أو 'user' (افتراضي: 'group')
     */
    public function sendToGroup(
        int $groupId,
        string $type,
        string $title,
        string $message,
        ?string $modelName = null,
        ?int $modelId = null,
        array $data = [],
        string $pushChannel = 'group'
    ): bool {
        // جلب جميع المستخدمين في الـ group
        $users = $this->getUsersByGroup($groupId);

        if ($users->isEmpty()) {
            $this->setError(['group' => ['No users found in the specified group']]);

            return false;
        }

        $senderBy = auth()->id();

        // حفظ إشعار في DB لكل user (Laravel Notifications) - استبعاد المرسل
        foreach ($users as $user) {
            // لا نرسل إشعار للمرسل نفسه
            if ($user->id == $senderBy) {
                continue;
            }

            $notification = new GenericNotification(
                title: $title,
                message: $message,
                type: $type,
                modelName: $modelName,
                modelId: $modelId,
                senderBy: $senderBy,
                data: $data,
                channel: null,
                expiresAt: null
            );

            $user->notify($notification);
        }

        // إرسال Push على Group Channel (channel واحد للجميع)
        if ($pushChannel === 'group') {
            event(new \App\Events\GroupNotificationEvent(
                title: $title,
                message: $message,
                type: $type,
                modelName: $modelName,
                modelId: $modelId,
                senderBy: $senderBy,
                data: $data,
                groupId: $groupId
            ));
        }

        return true;
    }

    /**
     * إرسال إشعار لمستخدمين محددين
     * يحفظ إشعار منفصل في DB لكل user + يرسل push على User channels
     *
     * @param  array  $userIds  Array of User IDs
     * @param  string  $type  نوع الإشعار
     * @param  string  $title  عنوان الإشعار
     * @param  string  $message  نص الإشعار
     * @param  string|null  $modelName  اسم الموديل
     * @param  int|null  $modelId  ID الموديل
     * @param  array  $data  بيانات إضافية
     */
    public function sendToUsers(
        array $userIds,
        string $type,
        string $title,
        string $message,
        ?string $modelName = null,
        ?int $modelId = null,
        array $data = []
    ): bool {
        $senderBy = auth()->id();

        foreach ($userIds as $userId) {
            // لا نرسل إشعار للمرسل نفسه
            if ($userId == $senderBy) {
                continue;
            }

            $user = User::find($userId);
            if (! $user) {
                continue;
            }

            $notification = new GenericNotification(
                title: $title,
                message: $message,
                type: $type,
                modelName: $modelName,
                modelId: $modelId,
                senderBy: $senderBy,
                data: $data,
                channel: null,
                expiresAt: null
            );

            $user->notify($notification);
        }

        return true;
    }

    /**
     * جلب جميع المستخدمين في Group
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUsersByGroup(int $groupId)
    {
        return User::whereHas('groups', function ($query) use ($groupId) {
            $query->where('groups.id', $groupId);
        })->get();
    }

    /**
     * جلب اسم المرسل
     */
    protected function getSenderName(?int $userId): ?string
    {
        if (! $userId) {
            return null;
        }

        $user = User::find($userId);

        return $user?->name;
    }
}
