<?php

namespace App\sys\Services\Profile;

use App\Models\Accommodation\AccommodationsCurrencies;
use App\Models\General\Currency;
use App\Models\General\TaxRateMappings;
use App\Models\Profile\AccommodationReservation;
use App\Models\Profile\AccommodationReservationRoom;
use App\Models\Profile\AccommodationReservationRoomTax;
use App\Models\Suppliers\SupplierAccountMappings;
use App\sys\Enums\SupplierAccountMappings as SupplierAccountMappingsEnum;
use App\sys\Helper;
use App\sys\Repository\Accounting\ConstraintRepository;
use App\sys\Repository\Profile\AccommodationReservationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

class AccommodationReservationService extends Services
{
    protected AccommodationReservationRepository $reservationRepository;

    public function __construct(AccommodationReservationRepository $reservationRepository)
    {
        $this->reservationRepository = $reservationRepository;
    }

    public function getPaginated(array $filters = [])
    {
        return $this->reservationRepository->getPaginated($filters);
    }

    public function getById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->findByIdOrFail($id);
    }

    public function create(array $request)
    {
        $financial = $this->authorizeFinancial('add');
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'accommodation_id' => ['required', 'integer', 'exists:ac_accommodations,id'],
            // 'type' => ['required', 'in:local,international'],
            // 'status' => ['required', 'in:pending,confirmed,cancelled'],
            'confirmation_date' => ['nullable', 'date'],
            'confirmation_num' => ['nullable', 'string', 'max:100'],
            'supplier_id' => [$financial, 'integer', 'exists:su_supplier,id'],
            // 'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'city_id' => ['required', 'integer', 'exists:cities,id'],
            'country_id' => ['required', 'integer', 'exists:countries,id'],
            'customer_note' => ['nullable', 'string'],
            'hotel_note' => ['nullable', 'string'],
            'reservation_num' => ['sometimes', 'nullable'],
            'reservation_link' => ['sometimes', 'nullable'],
            'count_rooms' => ['nullable'],
            'payment_date' => [$financial, 'date'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $reservation = $this->reservationRepository->create($request);

        // Update InvoiceServices next_pay_date if payment_date is provided
        if (isset($request['payment_date']) && $request['payment_date']) {
            $this->updateInvoiceServicesPaymentDate($reservation->id, $request['payment_date']);
        }

        return $reservation;
    }

    public function update(array $request)
    {
        $financial = $this->authorizeFinancial('edit');
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
            'profile_id' => ['sometimes', 'integer', 'exists:pr_profile,id'],
            'accommodation_id' => ['sometimes', 'integer', 'exists:ac_accommodations,id'],
            // 'type' => ['sometimes', 'in:local,international'],
            'status' => ['sometimes', 'in:pending,confirmed,cancelled,done'],
            'confirmation_date' => ['sometimes', 'nullable', 'date'],
            'confirmation_num' => ['sometimes', 'nullable', 'string', 'max:100'],
            'supplier_id' => ['sometimes', $financial, 'integer', 'exists:su_supplier,id'],
            // 'currency_id' => ['sometimes', $financial, 'integer', 'exists:currencies,id'],
            'city_id' => ['sometimes', 'integer', 'exists:cities,id'],
            'country_id' => ['sometimes', 'integer', 'exists:countries,id'],
            'customer_note' => ['sometimes', 'nullable', 'string'],
            'hotel_note' => ['sometimes', 'nullable', 'string'],
            'reservation_num' => ['sometimes', 'nullable'],
            'reservation_link' => ['sometimes', 'nullable'],
            'count_rooms' => ['nullable'],
            'payment_date' => ['sometimes', $financial, 'date'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $item = $this->reservationRepository->findByIdOrFail($request['id']);

        $reservation = $this->reservationRepository->update($item, $request);

        // Update InvoiceServices next_pay_date if payment_date is provided
        if (isset($request['payment_date']) && $request['payment_date']) {
            $this->updateInvoiceServicesPaymentDate($reservation->id, $request['payment_date']);
        }

        return $reservation;
    }

    private function authorizeFinancial($mode)
    {
        $ability = $mode == 'edit' ? 'EDIT_ROOM_COST' : 'ADD_ROOM_COST';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->del($ids);
    }

    public function getByAccommodationType(string $type)
    {
        $rules = [
            'type' => ['required', 'in:hotel,cruise'],
        ];
        $validator = Validator::make(['type' => $type], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getByAccommodationType($type);
    }

    public function getByAccommodationTypeAndProfile(string $type, int $profileId)
    {
        $rules = [
            'type' => ['required', 'in:hotel,cruise'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['type' => $type, 'profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getByAccommodationTypeAndProfile($type, $profileId);
    }

    public function reservationAttachments(array $request)
    {
        if (isset($request['reservation_id']) && ! empty($request['reservation_id'])) {
            $request['attachable_type'] = (new AccommodationReservation)->getMorphClass();
            $request['attachable_id'] = (int) $request['reservation_id'];
        }

        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'attachable_type' => ['required', 'string'],
            'attachable_id' => ['required', 'integer'],
            'attachment' => ['required', 'string'],
            'name' => ['sometimes', 'nullable', 'string'],
        ];

        if (! empty($request['attachable_type']) && class_exists($request['attachable_type'])) {
            $className = $request['attachable_type'];
            $request['attachable_type'] = (new $className)->getMorphClass();
        }

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $saveAttachment = Helper::saveFiles($request['attachment'], 'upload/reservation');

        if (! $saveAttachment['status']) {
            $this->setError(['attachment' => $saveAttachment['errors']]);

            return false;
        }
        $request['path'] = $saveAttachment['path'];

        // Default name to null if not provided
        if (! array_key_exists('name', $request)) {
            $request['name'] = null;
        }

        return $this->reservationRepository->uploadAttachments($request);
    }

    public function deleteAttachment($request)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_attachments,id'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        // Unlink files from disk first
        $attachments = $this->reservationRepository->findAttachmentsByIds($request['ids']);
        foreach ($attachments as $attachment) {
            $fullPath = public_path($attachment->path);
            if (file_exists($fullPath)) {
                @unlink($fullPath);
            }
        }

        return $this->reservationRepository->deleteAttachments($request['ids']);
    }

    public function getAttachmentsByProfileId(int $profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getAttachmentsByProfileId($profileId);
    }

    public function getAttachmentsByReservationId(int $reservationId)
    {
        $rules = [
            'reservation_id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['reservation_id' => $reservationId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getAttachmentsByReservationId($reservationId);
    }

    public function updateStatus(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
            'status' => ['required', 'in:pending,confirmed,cancelled,done'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $reservation = $this->reservationRepository->findByIdOrFail($request['id']);
        if (! $reservation) {
            $this->setError(['reservation' => ['الحجز غير موجود']]);

            return false;
        }

        $currentStatus = $reservation->status;
        $newStatus = $request['status'];

        // التحقق من قواعد تغيير الحالة
        if (! $this->canChangeStatus($currentStatus, $newStatus)) {
            $this->setError(['status' => ['لا يمكن تغيير الحالة من '.$currentStatus.' إلى '.$newStatus]]);

            return false;
        }

        // إذا كان التحويل إلى confirmed، التحقق من وجود supplier_id وإنشاء القيد
        if ($newStatus === 'confirmed') {
            if (! $reservation->supplier_id) {
                $this->setError(['supplier_id' => ['يجب تحديد المورد قبل تأكيد الحجز']]);

                return false;
            }

            // إنشاء القيد المحاسبي
            if (! $this->createSupplierPurchaseConstraint($reservation)) {
                return false;
            }
        }

        // تحديث الحالة
        $reservation->status = $newStatus;
        $reservation->save();

        return $reservation->load(['profile', 'accommodation']);
    }

    private function canChangeStatus(string $currentStatus, string $newStatus): bool
    {
        // done و cancelled لا يمكن تغييرهم
        if (in_array($currentStatus, ['done', 'cancelled'])) {
            return false;
        }

        // pending يمكن التحويل إلى confirmed أو cancelled فقط
        if ($currentStatus === 'pending') {
            return in_array($newStatus, ['confirmed', 'cancelled']);
        }

        // confirmed يمكن التحويل إلى cancelled أو done فقط (لا يمكن الرجوع لـ pending)
        if ($currentStatus === 'confirmed') {
            return in_array($newStatus, ['cancelled', 'done']);
        }

        return false;
    }

    // not used now
    private function createSupplierPurchaseConstraint(AccommodationReservation $reservation): bool
    {
        $defaultCurrency = Currency::where('is_default', 1)->first();
        if (! $defaultCurrency) {
            $this->setError(['currency_id' => ['الرجاء تعيين العملة الافتراضية للنظام.']]);

            return false;
        }

        // الحصول على الغرف مجمعة حسب العملة
        $rooms = AccommodationReservationRoom::selectRaw('currency_id, SUM(total_amount) as total_amount_sum')
            ->where('reservation_id', $reservation->id)
            ->groupBy('currency_id')
            ->get();

        if ($rooms->isEmpty()) {
            return true;
        }

        // الحصول على الضرائب مجمعة حسب العملة
        $taxAggregates = AccommodationReservationRoomTax::query()
            ->join('pr_accommodation_reservation_rooms as arr', 'arr.id', '=', 'pr_accommodation_reservation_room_tax.accommodation_reservation_rooms_id')
            ->selectRaw('arr.currency_id, pr_accommodation_reservation_room_tax.tax_rate_id, SUM(pr_accommodation_reservation_room_tax.tax_amount) as tax_sum')
            ->where('pr_accommodation_reservation_room_tax.accommodation_reservation_id', $reservation->id)
            ->groupBy('arr.currency_id', 'pr_accommodation_reservation_room_tax.tax_rate_id')
            ->get()
            ->groupBy('currency_id');

        $constraintRepo = new ConstraintRepository;
        $transfers = [];

        foreach ($rooms as $room) {
            $currencyId = (int) $room->currency_id;
            // accommodation_id يكون واحد فقط على مستوى الحجز بالكامل
            $accommodationId = (int) $reservation->accommodation_id;
            $totalAmount = (float) ($room->total_amount_sum ?? 0);

            if ($totalAmount <= 0) {
                continue;
            }

            $currency = Currency::find($currencyId);
            if (! $currency) {
                $this->setError(['currency_id' => ['لم يتم العثور على العملة المحددة للقيد.']]);

                return false;
            }

            $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);

            // الحصول على حساب تكلفة الخدمة من ac_accommodations_currencies
            $accommodationCurrency = AccommodationsCurrencies::where('accommodation_id', $accommodationId)
                ->where('currency_id', $currencyId)
                ->first();

            if (! $accommodationCurrency || ! $accommodationCurrency->tree_accounting_id) {
                $this->setError(['accounting' => ['برجاء تعيين حساب تكلفة الخدمة للإقامة للعملة '.$currency->code]]);

                return false;
            }

            // الحصول على حساب المورد
            $supplierAccount = SupplierAccountMappings::where('supplier_id', $reservation->supplier_id)
                ->where('currency_id', $currencyId)
                ->where('type', SupplierAccountMappingsEnum::START_BALANCE->value)
                ->first();

            if (! $supplierAccount || ! $supplierAccount->tree_account_id) {
                $this->setError(['accounting' => ['برجاء تعيين حساب آجل المورد للعملة '.$currency->code]]);

                return false;
            }

            // حساب الضرائب
            $taxRows = $taxAggregates->get($currencyId, collect());
            $taxTotal = (float) $taxRows->sum('tax_sum');

            // حساب صافي التكلفة (total_amount - taxes)
            $netCost = max(0, $totalAmount - $taxTotal);

            $currencyTransfers = [];

            // 1. مدين: تكلفة الخدمة
            $costDebitDefault = round($netCost * $exchangeRate, 2);
            $currencyTransfers[] = [
                'tree_accounting_id' => $accommodationCurrency->tree_accounting_id,
                'currency_id' => $currencyId,
                'name' => 'تكلفة خدمة الإقامة - حجز رقم '.$reservation->id,
                'debit' => $costDebitDefault,
                'creditor' => 0,
                'currency_debit' => round($netCost, 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ];

            $totalDebitDefault = $costDebitDefault;

            // 2. مدين: حسابات الضرائب
            foreach ($taxRows as $taxRow) {
                $taxAmount = (float) ($taxRow->tax_sum ?? 0);
                if ($taxAmount <= 0) {
                    continue;
                }

                $taxMapping = TaxRateMappings::where('tax_rate_id', $taxRow->tax_rate_id)
                    ->where('currency_id', $currencyId)
                    ->where('type', 'sales')
                    ->first();

                if (! $taxMapping || ! $taxMapping->tree_account_id) {
                    $this->setError(['tax' => ['برجاء تعيين حساب ضريبة المبيعات للضريبة رقم '.$taxRow->tax_rate_id.' للعملة '.$currency->code]]);

                    return false;
                }

                $taxDefault = round($taxAmount * $exchangeRate, 2);
                $totalDebitDefault += $taxDefault;

                $currencyTransfers[] = [
                    'tree_accounting_id' => $taxMapping->tree_account_id,
                    'currency_id' => $currencyId,
                    'name' => 'ضريبة مبيعات - حجز رقم '.$reservation->id,
                    'debit' => $taxDefault,
                    'creditor' => 0,
                    'currency_debit' => round($taxAmount, 2),
                    'currency_creditor' => 0,
                    'cost_center_id' => null,
                    'currency_transfer_rate' => $exchangeRate,
                ];
            }

            // 3. دائن: حساب المورد
            $supplierCreditDefault = round($totalAmount * $exchangeRate, 2);
            $currencyTransfers[] = [
                'tree_accounting_id' => $supplierAccount->tree_account_id,
                'currency_id' => $currencyId,
                'name' => 'آجل مورد - حجز رقم '.$reservation->id,
                'debit' => 0,
                'creditor' => $supplierCreditDefault,
                'currency_debit' => 0,
                'currency_creditor' => round($totalAmount, 2),
                'cost_center_id' => null,
                'currency_transfer_rate' => $exchangeRate,
            ];

            // التحقق من توازن القيد
            $difference = $totalDebitDefault - $supplierCreditDefault;
            if (abs($difference) > 0.02) {
                $this->setError(['constraint' => ['قيمة القيد غير متوازنة للعملة '.$currency->code]]);

                return false;
            }

            // إضافة الفرق البسيط للدائن إذا لزم الأمر
            if (abs($difference) > 0) {
                $lastIndex = count($currencyTransfers) - 1;
                $currencyTransfers[$lastIndex]['creditor'] = round($currencyTransfers[$lastIndex]['creditor'] + $difference, 2);
            }

            foreach ($currencyTransfers as $transfer) {
                $transfers[] = $transfer;
            }
        }

        if (empty($transfers)) {
            return true;
        }

        $totalDebitDefault = array_sum(array_map(static fn ($transfer) => $transfer['debit'], $transfers));
        $totalCreditDefault = array_sum(array_map(static fn ($transfer) => $transfer['creditor'], $transfers));

        $profile = $reservation->profile;

        $constraintData = [
            'customer_id' => $profile->customer_id ?? null,
            'profile_id' => $profile->id,
            'supplier_id' => $reservation->supplier_id,
            'sales_id' => $profile->user_id ?? null,
            'date' => now()->format('Y-m-d'),
            'total_creditor' => $totalCreditDefault,
            'total_debit' => $totalDebitDefault,
            'name' => 'قيد استلام الخدمة من المورد (فاتورة شراء)',
            'description' => 'قيد استلام الخدمة من المورد - حجز رقم '.$reservation->id,
            'active' => 1,
            'capture_exchange' => 'constraint',
            'creation_mode' => 'automatic',
            'type_optional' => null,
            'currency_id' => null,
            'currency_transfer_rate' => 1,
            'company_id' => $profile->company_id,
            'transfers' => $transfers,
        ];

        $constraintRepo->create($constraintData, 'yes');

        return true;
    }

    /**
     * Update next_pay_date for all InvoiceServices related to this reservation
     */
    private function updateInvoiceServicesPaymentDate(int $reservationId, string $paymentDate): void
    {
        \App\Models\invoice\InvoiceServices::where('reservation_id', $reservationId)
            ->update(['next_pay_date' => $paymentDate]);
    }
}
