<?php

namespace App\sys\Services\User;

use App\Models\Accounting\Paytype;
use App\Models\UserAccountMapping;
use App\sys\Enums\PaymentTypes;
use App\sys\Repository\User\UserRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class UserService extends Services
{
    protected $userrepo;

    public function __construct(UserRepository $userrepo)
    {
        $this->userrepo = $userrepo;

    }

    public function getPaginated()
    {
        // نستدعي الريبوزيتوري مباشرة
        return $this->userrepo->getPaginated();

    }

    public function get($id)
    {

        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:users,id'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($id);

        return $user;
    }

    public function create(array $data)
    {

        $rules = [
            'name' => 'required|string|max:255',
            'first_name' => 'nullable|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'is_active' => 'boolean',
            'employee_number' => 'nullable|string|max:50',

            'role_id' => 'required|integer|exists:roles,id',
            'is_supplier' => 'sometimes|boolean',

            'groups' => 'nullable|array',
            'groups.*' => 'integer|exists:groups,id',

            'services' => 'nullable|array',
            'services.*' => 'integer|exists:services,id',

            'accommodations' => 'nullable|array',
            'accommodations.*' => 'integer|exists:ac_accommodations,id',

            'companies' => 'nullable|array',
            'companies.*' => 'integer|exists:companies,id',

            'accounting' => 'required|array|min:1',
            'accounting.*.currency_id' => 'required|integer|exists:currencies,id',
            'accounting.*.percentage' => 'required|numeric|min:0|max:100',
            'accounting.*.commission_account_id' => 'required|integer|exists:tree_accounting,id',
            'accounting.*.wallet_account_id' => 'required|integer|exists:tree_accounting,id',
            'accounting.*.Supplier_account_id' => 'sometimes|integer|exists:tree_accounting,id',
        ];

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        try {
            return $this->userrepo->create($data);
        } catch (\Exception $e) {
            Log::error('Failed to create user: '.$e->getMessage());
            $this->setError(['database' => ['An error occurred while creating the user.']]);

            return false;
        }
    }

    public function update($id, $data)
    {
        $rules = [
            'name' => 'sometimes|string|max:255',
            'first_name' => 'sometimes|nullable|string|max:255',
            'last_name' => 'sometimes|nullable|string|max:255',
            'phone' => ['sometimes', 'string', Rule::unique('users')->ignore($id), 'max:50'],
            'email' => ['sometimes', 'email', Rule::unique('users')->ignore($id), 'max:255'],
            'password' => ['sometimes', 'nullable', 'confirmed', Password::min(8)],
            'is_active' => 'sometimes|boolean',
            'employee_number' => 'sometimes|nullable|string|max:100',
            'role_id' => 'sometimes|exists:roles,id',
            'is_supplier' => 'sometimes|boolean',

            'groups' => 'sometimes|array',
            'groups.*' => 'exists:groups,id',

            'services' => 'sometimes|array',
            'services.*' => 'exists:services,id',

            'accommodations' => 'sometimes|array',
            'accommodations.*' => 'exists:ac_accommodations,id',

            'companies' => 'sometimes|array',
            'companies.*' => 'exists:companies,id',

            'accounting' => 'sometimes|array',
            'accounting.*.currency_id' => 'sometimes|required|exists:currencies,id',
            'accounting.*.percentage' => 'sometimes|required|numeric|min:0|max:100',
            'accounting.*.commission_account_id' => 'sometimes|required|exists:tree_accounting,id',
            'accounting.*.wallet_account_id' => 'sometimes|required|exists:tree_accounting,id',
            'accounting.*.Supplier_account_id' => 'sometimes|exists:tree_accounting,id',
        ];

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        if (empty($data['password'])) {
            unset($data['password']);
        }

        return $this->userrepo->update($id, $data);

    }

    public function delete($id)
    {

        return $this->userrepo->delete($id);

    }

    public function updateOverrides(int $userId, array $overrides)
    {

        $permissionIds = array_keys($overrides);
        $validator = Validator::make([
            'user_id' => $userId,
            'permission_ids' => $permissionIds,
            'overrides' => $overrides,
        ], [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'permission_ids.*' => ['integer', 'exists:permissions,id'],
            'overrides.*.override_type' => ['required', Rule::in(['allow', 'deny'])],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($userId);
        if (! $user) {
            $this->setError(['user' => 'User not found']);

            return false;
        }

        $this->userrepo->syncOverrides($userId, $overrides);

        // $user->clearPermissionCache();

        return true;
    }

    public function updateUserRole(int $userId, ?int $roleId)
    {
        // 1. [جديد] - إضافة Validation
        $validator = Validator::make([
            'user_id' => $userId,
            'role_id' => $roleId,
        ], [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'role_id' => ['nullable', 'integer', 'exists:roles,id'], // 'nullable' عشان نسمح بإلغاء الـ Role
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $user = $this->userrepo->find($userId);
        if (! $user) {
            $this->setError(['user' => 'User not found']);

            return false;
        }

        $this->userrepo->setRole($userId, $roleId);

        // $user->clearPermissionCache();

        return true;
    }

    /**
     * Get wallet balance report for a user grouped by currency
     *
     * @return array|false
     */
    public function getWalletReport(int $userId)
    {
        $validator = Validator::make(['id' => $userId], [
            'id' => ['required', 'integer', 'exists:users,id'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $report = $this->userrepo->getWalletReport($userId);

        if ($report === false) {
            $this->setError(['wallet' => ['No wallet accounts found for this user']]);

            return false;
        }

        return $report;
    }

    /**
     * دفع أو أخذ من محفظة الموظف
     */
    public function wallet($data)
    {
        // 1. Validation
        $rules = [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'company_id' => ['required', 'integer', 'exists:companies,id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'amount' => ['required', 'numeric'],
            'type' => ['required', 'in:receipt,payment'],
            'pay_type_id' => ['required', 'exists:pay_type,id'],
            'accounting_id' => ['nullable', 'integer', 'exists:pay_type_accounting,id'],
            'description' => ['nullable', 'string'],
            'date' => ['required', 'date'],
        ];
        $validator = Validator::make($data, $rules);

        $payType = Paytype::find($data['pay_type_id']);
        if (in_array($payType->type, ['bank', 'payment_method_rates']) && $data['accounting_id'] == null) {
            $validator->errors()->add('accounting_id', 'It cannot be empty');
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // 2. المتغيرات الأساسية
        $company_id = $data['company_id'];
        $currency_id = $data['currency_id'];
        $amount = (float) $data['amount'];
        $pay_type_id = $data['pay_type_id'];
        $type = $data['type'];
        $user_id = $data['user_id'];
        $date = $data['date'];
        $accounting_id = $data['accounting_id'] ?? null;
        $description = $data['description'] ?? null;

        // 3. جلب حساب محفظة الموظف wallet_account_id من user_account_mappings
        $walletMapping = UserAccountMapping::where('user_id', $user_id)
            ->where('currency_id', $currency_id)
            ->first();

        if (! $walletMapping || ! $walletMapping->wallet_account_id) {
            $this->setError(['wallet_account' => 'حساب محفظة الموظف غير موجود لهذه العملة']);

            return false;
        }
        $wallet_account = $walletMapping->wallet_account_id;
        $transfer_account = null;
        $commission_account = null;
        $commission_percentage = 0;
        $commission_amount = 0;
        $paytypeType = $payType->type;

        // 4. جلب الحساب المقابل & العمولة
        if ($paytypeType === PaymentTypes::COMPANY_CASH->value) {
            $cash = \App\Models\General\CompanyAccountMappings::where([
                'company_id' => $company_id,
                'currency_id' => $currency_id,
                'type' => 'cach',
            ])->first();
            if (! $cash) {
                $this->setError(['fallback_account' => 'لم يتم العثور على حساب cash كـ fallback']);

                return false;
            }
            $transfer_account = $cash->tree_account_id;
        } elseif ($paytypeType === PaymentTypes::BANK->value) {
            $payTypeRow = \App\Models\Accounting\PaytypeAccounting::find($accounting_id);
            $transfer_account = $payTypeRow->tree_accounting_transfer_id;
        } elseif ($paytypeType === PaymentTypes::PAYMENT_METHOD_RATES->value) {
            $payTypeRow = \App\Models\Accounting\PaytypeAccounting::find($accounting_id);
            $transfer_account = $payTypeRow->tree_accounting_transfer_id;
            $commission_account = $payTypeRow->tree_account_commission_id;
            $commission_percentage = (float) $payTypeRow->percentage;
            if ($commission_account && $commission_percentage > 0) {
                $commission_amount = round(($amount * $commission_percentage) / 100, 2);
            }
        } elseif ($paytypeType === PaymentTypes::EMPLOYEE_CASH->value) {
            // في حالة employee_cash، نحتاج حساب محفظة موظف آخر
            $employeeWalletMapping = UserAccountMapping::where('user_id', Auth::id())
                ->where('currency_id', $data['currency_id'])
                ->first();

            if (! $employeeWalletMapping || ! $employeeWalletMapping->wallet_account_id) {
                $this->setError(['pay_type_id' => ['برجاء تعيين حساب المحفظة للموظف لهذه العملة.']]);

                return false;
            }
            $transfer_account = $employeeWalletMapping->wallet_account_id;
        }

        // 5. تجهيز ترتيب التحويلات (transfer array)
        $transfers = [];
        if ($type == 'receipt') {
            $transfers[] = [
                'tree_accounting_id' => $wallet_account,
                'name' => 'wallet',
                'debit' => $amount,
                'creditor' => 0,
            ];
            $transfers[] = [
                'tree_accounting_id' => $transfer_account,
                'name' => 'wallet counterparty',
                'debit' => 0,
                'creditor' => ($amount - $commission_amount),
            ];
            if ($commission_amount > 0 && $commission_account) {
                $transfers[] = [
                    'tree_accounting_id' => $commission_account,
                    'name' => 'wallet commission',
                    'debit' => 0,
                    'creditor' => $commission_amount,
                ];
            }
        } else { // payment
            $transfers[] = [
                'tree_accounting_id' => $wallet_account,
                'name' => 'wallet',
                'debit' => 0,
                'creditor' => $amount,
            ];
            $transfers[] = [
                'tree_accounting_id' => $transfer_account,
                'name' => 'wallet counterparty',
                'debit' => ($amount - $commission_amount),
                'creditor' => 0,
            ];
            if ($commission_amount > 0 && $commission_account) {
                $transfers[] = [
                    'tree_accounting_id' => $commission_account,
                    'name' => 'wallet commission',
                    'debit' => $commission_amount,
                    'creditor' => 0,
                ];
            }
        }

        // 6. تحويل العملة إذا العملة ليست الافتراضية
        $default_currency_id = \App\Models\General\Currency::where('is_default', 1)->value('id');
        $exchange_rate = 1;
        if ($currency_id != $default_currency_id) {
            $exchange_rate = (float) \App\Models\General\Currency::find($currency_id)->exchange_rate;
            foreach ($transfers as &$t) {
                if ($t['debit'] > 0) {
                    $t['debit'] = round($t['debit'] * $exchange_rate, 2);
                    $t['currency_debit'] = $t['debit'];
                }
                if ($t['creditor'] > 0) {
                    $t['creditor'] = round($t['creditor'] * $exchange_rate, 2);
                    $t['currency_creditor'] = $t['creditor'];
                }
            }
            unset($t);
        }

        // 7. فحص التوازن المحاسبي
        $total_debit = array_sum(array_column($transfers, 'debit'));
        $total_creditor = array_sum(array_column($transfers, 'creditor'));
        $currency_creditor = array_sum(array_column($transfers, 'currency_creditor'));
        $currency_debit = array_sum(array_column($transfers, 'currency_debit'));
        if (abs($total_debit - $total_creditor) > 0.01) {
            $this->setError(['balance' => 'عدد المدين لا يساوي الدائن']);

            return false;
        }

        // 8. تجهيز البيانات ثم الارسال
        $const = [
            'user_id_emp' => $user_id,
            'date' => $date,
            'total_creditor' => $total_creditor,
            'total_debit' => $total_debit,
            'currency_creditor' => $currency_creditor,
            'currency_debit' => $currency_debit,
            'payType_id' => $pay_type_id,
            'type_optional' => 'wallet',
            'name' => 'wallet',
            'description' => $description,
            'active' => 1,
            'capture_exchange' => $type,
            'currency_id' => $currency_id,
            'currency_transfer_rate' => $exchange_rate,
            'company_id' => $company_id,
            'transfers' => $transfers,
        ];

        // 9. التنفيذ
        return app(\App\sys\Repository\Accounting\ConstraintRepository::class)->create($const);
    }

    public function walletTransactions($data)
    {
        $rules = [
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'from' => ['sometimes', 'nullable', 'date'],
            'to' => ['sometimes', 'nullable', 'date'],
            'currency_id' => ['sometimes', 'nullable',  'exists:currencies,id'],
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $currency_id = $data['currency_id'] ?? null;
        $walletMapping = UserAccountMapping::where('user_id', $data['user_id'])
            ->when($currency_id, function ($query) use ($currency_id) {
                $query->where('currency_id', $currency_id);
            })->pluck('wallet_account_id')
            ->unique()
            ->values()
            ->all();

        if (empty($walletMapping)) {
            $validator->errors()->add('currency_id', 'لا يوجد حسابات لهذه العمله');
            $this->setError($validator->errors());

            return false;
        }

        return $this->userrepo->walletTransactions($data, $walletMapping);

    }
}
