<?php

namespace App\Http\Resources\Accounting;

use App\Http\Resources\CompanyResource;
use App\Http\Resources\Profile\CustomerResource;
use App\Http\Resources\Profile\ProfileResource;
use Illuminate\Http\Resources\Json\JsonResource;

class InvoiceResource extends JsonResource
{
    public function toArray($request)
    {
        return [
            'id' => $this->id,
            'serial_num' => $this->serial_num,
            'profile_num' => $this->profile->profile_number ?? null,
            'profile_id' => $this->profile_id,
            'profile_user_id' => $this->profile->user_id ?? null,
            'customer_id' => $this->customer_id,
            'company_id' => $this->company_id,
            'invoice_date' => $this->invoice_date,
            'recip_date' => $this->recip_date,
            'pay_duration' => $this->pay_duration,
            'invoice_date_due' => $this->invoice_date_due,
            'invoice_price' => $this->invoice_price,
            'invoice_type' => $this->invoice_type,
            'paid_price' => $this->paid_price,
            'paid_date' => $this->paid_date,
            'remaining_price' => $this->remaining_price,
            'worthy_days' => $this->worthy_days,
            'notes' => $this->notes,
            'conditions' => $this->conditions,
            'next_paid_date' => $this->next_paid_date,
            'status' => $this->getStatus(),
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            'profile' => $this->whenLoaded('profile', function () {
                return $this->profile ? new ProfileResource($this->profile) : null;
            }),
            'customer' => $this->whenLoaded('customer', function () {
                return $this->customer ? new CustomerResource($this->customer) : null;
            }),
            'company' => $this->whenLoaded('company', function () {
                return $this->company ? new CompanyResource($this->company) : null;
            }),
            'profile_user' => $this->when(
                $this->relationLoaded('profile') && $this->profile && $this->profile->relationLoaded('user'),
                function () {
                    return $this->profile->user ? [
                        'id' => $this->profile->user->id,
                        'name' => $this->profile->user->name,
                        'email' => $this->profile->user->email,
                    ] : null;
                }
            ),
        ];
    }

    private function getStatus(): string
    {
        $invoicePrice = (float) ($this->invoice_price ?? 0);
        $paidPrice = (float) ($this->paid_price ?? 0);
        $remainingPrice = (float) ($this->remaining_price ?? 0);

        // deferred: invoice_price = 0
        if ($invoicePrice == 0) {
            return 'deferred';
        }

        // paid: invoice_price = paid_price AND remaining_price = 0
        if (abs($invoicePrice - $paidPrice) < 0.01 && $remainingPrice == 0) {
            return 'paid';
        }

        // partiallyPaid: (paid_price < invoice_price AND paid_price > 0) OR (remaining_price < invoice_price AND remaining_price > 0)
        // But not paid
        if (($paidPrice > 0 && $paidPrice < $invoicePrice) || ($remainingPrice > 0 && $remainingPrice < $invoicePrice)) {
            return 'partiallyPaid';
        }

        // Default to deferred if no match
        return 'deferred';
    }
}
