<?php

namespace App\Http\Resources\Profile;

use Illuminate\Http\Resources\Json\JsonResource;

class CustomerProfileResource extends JsonResource
{
    public function toArray($request)
    {
        return [
            'profile_id' => $this['profile_id'] ?? null,
            'status' => $this['status'] ?? null,
            'link_code' => $this['link_code'] ?? null,
            'profile_number' => $this['profile_number'] ?? null,
            'customer_name' => $this['customer_name'] ?? null,
            'arrival_date' => $this['arrival_date'] ?? null,
            'departure_date' => $this['departure_date'] ?? null,
            'company_name' => $this['company_name'] ?? null,
            'currencies' => $this['currencies'] ?? null,
            'total_currencies' => count($this['currencies']),
            'summary' => $this->getSummary(),
        ];
    }

    /**
     * Get summary of all currencies
     */
    private function getSummary()
    {
        $currencies = collect($this['currencies']);

        if ($currencies->isEmpty()) {
            return [
                'total_cost' => 0,
                'total_paid' => 0,
                'total_remaining' => 0,
                'payment_percentage' => 0,
            ];
        }

        // Use the configured default currency rates to normalize totals before summing
        $defaultCurrency = $currencies->firstWhere('is_default', true) ?? $currencies->first();
        $defaultRate = (float) ($defaultCurrency['exchange_rate'] ?? 1) ?: 1;

        $totals = $currencies->reduce(function ($carry, $currency) use ($defaultRate) {
            $exchangeRate = (float) ($currency['exchange_rate'] ?? 1);
            $rateToDefault = $defaultRate > 0 ? $exchangeRate / $defaultRate : 1;

            $carry['cost'] += ((float) ($currency['total_cost'] ?? 0)) * $rateToDefault;
            $carry['paid'] += ((float) ($currency['total_paid'] ?? 0)) * $rateToDefault;
            $carry['remaining'] += ((float) ($currency['remaining'] ?? 0)) * $rateToDefault;

            return $carry;
        }, ['cost' => 0, 'paid' => 0, 'remaining' => 0]);

        return [
            'total_cost' => round($totals['cost'], 2),
            'total_paid' => round($totals['paid'], 2),
            'total_remaining' => round($totals['remaining'], 2),
            'payment_percentage' => $totals['cost'] > 0 ? round(($totals['paid'] / $totals['cost']) * 100, 2) : 0,
        ];
    }
}
